# Project Management Backend API

A comprehensive Node.js backend system with SQL database integration for project management applications, featuring phone number + OTP authentication, user management, and RESTful APIs.

> **Note**: This application has been migrated from SQLite to MySQL. See [MIGRATION_GUIDE.md](MIGRATION_GUIDE.md) for details.

## Features

- 📱 **Phone Number + OTP Authentication** - Secure login with test credentials
- 👥 **User Management** - Admin functionality for sub-admin/company owner accounts
- 🏗️ **Project Management** - Complete CRUD operations for projects
- 🏃‍♂️ **Sprint Management** - Agile sprint planning and tracking
- ✅ **Task Management** - Task creation, assignment, and tracking
- 🔐 **Role-based Access Control** - Admin, Sub-admin, and User roles
- 📊 **Activity Logging** - Complete audit trail
- 🚀 **RESTful APIs** - Well-documented API endpoints

## Tech Stack

- **Runtime**: Node.js
- **Framework**: Express.js
- **Database**: MySQL
- **Authentication**: JWT + OTP
- **Validation**: express-validator
- **Security**: helmet, bcrypt, rate limiting
- **Documentation**: Postman collection included

## Quick Start

### Prerequisites

- Node.js (v14 or higher)
- MySQL (v8.0 or higher)
- npm or yarn

> **SQLite Migration**: If you're migrating from SQLite, see [MIGRATION_GUIDE.md](MIGRATION_GUIDE.md) for detailed instructions.

### Installation

1. **Clone and navigate to the backend directory**
   ```bash
   cd backend
   ```

2. **Install dependencies**
   ```bash
   npm install
   ```

3. **Set up environment variables**
   ```bash
   cp .env.example .env
   ```
   
   Update the `.env` file with your configuration:
   ```env
   # Server Configuration
   PORT=3000
   NODE_ENV=development
   
   # Database Configuration
   DB_HOST=localhost
   DB_PORT=3306
   DB_USER=your_mysql_user
   DB_PASSWORD=your_mysql_password
   DB_NAME=project_management
   
   # JWT Configuration
   JWT_SECRET=your_super_secret_jwt_key_here
   JWT_EXPIRES_IN=1h
   
   # OTP Configuration
   OTP_EXPIRY_MINUTES=5
   TEST_PHONE=12567890
   
   # Twilio Configuration (for production OTP)
   TWILIO_ACCOUNT_SID=your_twilio_account_sid
   TWILIO_AUTH_TOKEN=your_twilio_auth_token
   TWILIO_PHONE_NUMBER=your_twilio_phone_number
   ```

4. **Set up MySQL database**
   
   Create a MySQL database:
   ```sql
   CREATE DATABASE project_management;
   ```

5. **Run database migration**
   ```bash
   node scripts/migrate.js
   ```

6. **Start the server**
   ```bash
   npm start
   ```
   
   For development with auto-reload:
   ```bash
   npm run dev
   ```

The server will start on `http://localhost:3000`

## Test Credentials

For testing the authentication system:
- **Phone Number**: `12567890`
- **OTP**: `123456`

## API Documentation

### Base URL
```
http://localhost:3000
```

### Authentication Flow

1. **Send OTP**
   ```http
   POST /api/auth/send-otp
   Content-Type: application/json
   
   {
     "phone_number": "12567890"
   }
   ```

2. **Verify OTP & Login**
   ```http
   POST /api/auth/verify-otp
   Content-Type: application/json
   
   {
     "phone_number": "12567890",
     "otp": "123456"
   }
   ```

3. **Use the returned access token in subsequent requests**
   ```http
   Authorization: Bearer <access_token>
   ```

### API Endpoints

#### Authentication
- `POST /api/auth/send-otp` - Send OTP to phone number
- `POST /api/auth/verify-otp` - Verify OTP and login
- `POST /api/auth/refresh` - Refresh access token
- `GET /api/auth/profile` - Get user profile
- `GET /api/auth/verify` - Verify token validity
- `POST /api/auth/logout` - Logout from current device
- `POST /api/auth/logout-all` - Logout from all devices

#### User Management (Admin only)
- `GET /api/users` - Get all users with pagination
- `GET /api/users/:id` - Get user by ID
- `POST /api/users` - Create new user
- `PUT /api/users/:id` - Update user
- `DELETE /api/users/:id` - Delete user (Super Admin only)
- `GET /api/users/stats` - Get user statistics

#### Projects
- `GET /api/projects` - Get all projects
- `GET /api/projects/:id` - Get project by ID
- `POST /api/projects` - Create project (Admin only)
- `PUT /api/projects/:id` - Update project (Admin only)
- `DELETE /api/projects/:id` - Delete project (Admin only)
- `POST /api/projects/:id/members` - Add project member (Admin only)
- `DELETE /api/projects/:id/members/:userId` - Remove project member (Admin only)

#### Sprints
- `GET /api/sprints` - Get all sprints
- `GET /api/sprints/:id` - Get sprint by ID
- `POST /api/sprints` - Create sprint
- `PUT /api/sprints/:id` - Update sprint
- `DELETE /api/sprints/:id` - Delete sprint (Admin only)
- `POST /api/sprints/:id/start` - Start sprint
- `POST /api/sprints/:id/complete` - Complete sprint

#### Tasks
- `GET /api/tasks` - Get all tasks
- `GET /api/tasks/:id` - Get task by ID
- `POST /api/tasks` - Create task
- `PUT /api/tasks/:id` - Update task
- `DELETE /api/tasks/:id` - Delete task
- `POST /api/tasks/:id/comments` - Add task comment

## Postman Collection

Import the Postman collection from `postman/Project_Management_API.postman_collection.json` to test all API endpoints.

The collection includes:
- Pre-configured environment variables
- Automatic token management
- Test scripts for response validation
- Complete API coverage

## Database Schema

The system includes the following main entities:

- **users** - User accounts with roles and authentication
- **otps** - OTP verification codes
- **refresh_tokens** - JWT refresh tokens
- **projects** - Project information and management
- **project_members** - Project team membership
- **sprints** - Sprint planning and tracking
- **tasks** - Task management and assignment
- **task_comments** - Task discussion threads
- **task_attachments** - File attachments for tasks
- **activity_logs** - System activity audit trail

## User Roles

1. **Admin** - Full system access, can manage all users and projects
2. **Sub-admin** - Can manage projects and users (limited)
3. **User** - Can access assigned projects and tasks

## Security Features

- JWT-based authentication with refresh tokens
- Phone number + OTP verification
- Password hashing with bcrypt
- Rate limiting on API endpoints
- Input validation and sanitization
- SQL injection prevention
- CORS protection
- Security headers with helmet

## Development

### Running Tests
```bash
npm test
```

### Migration Scripts

For SQLite to MySQL migration:
```bash
# Test MySQL connection
npm run test-mysql

# Migrate data from SQLite to MySQL
npm run migrate-sqlite-to-mysql

# Test the migration process
npm run test-migration
```

### Code Structure
```
backend/
├── config/          # Database and configuration
├── middleware/      # Authentication and validation
├── routes/          # API route handlers
├── services/        # Business logic
├── scripts/         # Database migration and utilities
├── postman/         # API documentation
└── server.js        # Application entry point
```

### Environment Variables

All configuration is managed through environment variables. See `.env` file for complete list.

## Production Deployment

1. Set `NODE_ENV=production`
2. Configure production database
3. Set up Twilio for real OTP sending
4. Configure proper JWT secrets
5. Set up SSL/TLS certificates
6. Configure reverse proxy (nginx)
7. Set up process manager (PM2)

## Default Admin Account

After running the migration, a default admin account is created:
- **Phone**: `12567890`
- **Role**: `admin`
- **Status**: `active`

Use the test OTP `123456` to login with this account.

## Support

For issues and questions, please check the API responses for detailed error messages. All endpoints return consistent JSON responses with success/error indicators.

## License

This project is part of a Flutter project management application.