const sqlite3 = require('sqlite3').verbose();
const path = require('path');

// Database file path
const dbPath = path.join(__dirname, '..', 'database.sqlite');

let db = null;

const initDatabase = () => {
  return new Promise((resolve, reject) => {
    db = new sqlite3.Database(dbPath, (err) => {
      if (err) {
        console.error('❌ Error opening SQLite database:', err.message);
        reject(err);
      } else {
        console.log('✅ Connected to SQLite database');
        createTables().then(resolve).catch(reject);
      }
    });
  });
};

const resetDatabase = () => {
  return new Promise((resolve, reject) => {
    if (!db) {
      reject(new Error('Database not initialized'));
      return;
    }

    // Drop all existing tables to reset the database
    const dropTablesSQL = `
      DROP TABLE IF EXISTS task_assignments;
      DROP TABLE IF EXISTS tasks;
      DROP TABLE IF EXISTS activity_logs;
      DROP TABLE IF EXISTS project_permissions;
      DROP TABLE IF EXISTS project_members;
      DROP TABLE IF EXISTS projects;
      DROP TABLE IF EXISTS user_permissions;
      DROP TABLE IF EXISTS company_users;
      DROP TABLE IF EXISTS companies;
      DROP TABLE IF EXISTS refresh_tokens;
      DROP TABLE IF EXISTS otps;
      DROP TABLE IF EXISTS users;
      DROP TABLE IF EXISTS sprints;
    `;

    db.exec(dropTablesSQL, (err) => {
      if (err) {
        console.error('❌ Error dropping tables:', err.message);
        reject(err);
      } else {
        console.log('✅ All tables dropped successfully');
        createTables().then(resolve).catch(reject);
      }
    });
  });
};

const createTables = () => {
  return new Promise((resolve, reject) => {
    const createTablesSQL = `
      -- Companies table for hierarchical organization
      CREATE TABLE IF NOT EXISTS companies (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT UNIQUE NOT NULL,
        description TEXT,
        admin_phone TEXT NOT NULL,
        status TEXT DEFAULT 'active' CHECK(status IN ('active', 'inactive')),
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
      );

      -- Users table with simplified role model
      CREATE TABLE IF NOT EXISTS users (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        username TEXT UNIQUE NOT NULL,
        phone_number TEXT UNIQUE NOT NULL,
        first_name TEXT,
        last_name TEXT,
        role TEXT DEFAULT 'user' CHECK(role IN ('main_admin', 'admin', 'user')),
        permission_level INTEGER DEFAULT 1 CHECK(permission_level IN (1, 2, 3)),
        company_id INTEGER,
        status TEXT DEFAULT 'active' CHECK(status IN ('active', 'inactive', 'suspended')),
        profile_image TEXT,
        created_by INTEGER,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (company_id) REFERENCES companies(id) ON DELETE SET NULL,
        FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL
      );

      -- Company users relationship table
      CREATE TABLE IF NOT EXISTS company_users (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        company_id INTEGER NOT NULL,
        user_id INTEGER NOT NULL,
        role TEXT DEFAULT 'user' CHECK(role IN ('admin', 'user')),
        permissions TEXT, -- JSON string for flexible permissions
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (company_id) REFERENCES companies(id) ON DELETE CASCADE,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        UNIQUE(company_id, user_id)
      );

      -- OTP table for authentication
      CREATE TABLE IF NOT EXISTS otps (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        phone_number TEXT NOT NULL,
        otp_code TEXT NOT NULL,
        expires_at DATETIME NOT NULL,
        is_used BOOLEAN DEFAULT FALSE,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
      );

      -- Refresh tokens table
      CREATE TABLE IF NOT EXISTS refresh_tokens (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER NOT NULL,
        token TEXT NOT NULL,
        expires_at DATETIME NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
      );

      -- Projects table
      CREATE TABLE IF NOT EXISTS projects (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT UNIQUE NOT NULL,
        description TEXT,
        company_id INTEGER,
        status TEXT DEFAULT 'planning' CHECK(status IN ('planning', 'active', 'on_hold', 'completed', 'cancelled')),
        priority TEXT DEFAULT 'medium' CHECK(priority IN ('low', 'medium', 'high', 'critical')),
        start_date DATE,
        end_date DATE,
        budget DECIMAL(15,2),
        owner_id INTEGER NOT NULL,
        created_by INTEGER NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (company_id) REFERENCES companies(id) ON DELETE SET NULL,
        FOREIGN KEY (owner_id) REFERENCES users(id) ON DELETE CASCADE,
        FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE CASCADE
      );

      -- Project members table
      CREATE TABLE IF NOT EXISTS project_members (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        project_id INTEGER NOT NULL,
        user_id INTEGER NOT NULL,
        role TEXT DEFAULT 'user' CHECK(role IN ('main_admin', 'admin', 'user')),
        permissions TEXT, -- JSON string for project-specific permissions
        joined_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        assigned_by INTEGER,
        FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE CASCADE,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        FOREIGN KEY (assigned_by) REFERENCES users(id) ON DELETE SET NULL,
        UNIQUE(project_id, user_id)
      );

      -- Project permissions table for granular access control
      CREATE TABLE IF NOT EXISTS project_permissions (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER NOT NULL,
        project_id INTEGER NOT NULL,
        can_view BOOLEAN DEFAULT TRUE,
        can_create_tasks BOOLEAN DEFAULT FALSE,
        can_edit_tasks BOOLEAN DEFAULT FALSE,
        can_delete_tasks BOOLEAN DEFAULT FALSE,
        can_assign_tasks BOOLEAN DEFAULT FALSE,
        can_manage_members BOOLEAN DEFAULT FALSE,
        created_by INTEGER,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE CASCADE,
        FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL,
        UNIQUE(user_id, project_id)
      );

      -- Tasks table with comprehensive fields
      CREATE TABLE IF NOT EXISTS tasks (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        title TEXT NOT NULL,
        description TEXT NOT NULL,
        project_id INTEGER NOT NULL,
        assigned_to INTEGER,
        status TEXT DEFAULT 'To Do' CHECK(status IN ('To Do', 'In Progress', 'Review', 'Done', 'Cancelled')),
        priority TEXT DEFAULT 'Low' CHECK(priority IN ('Low', 'Medium', 'High', 'Critical')),
        due_date DATETIME,
        estimated_hours DECIMAL(5,2),
        actual_hours DECIMAL(5,2),
        created_by INTEGER NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        completed_at DATETIME,
        FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE CASCADE,
        FOREIGN KEY (assigned_to) REFERENCES users(id) ON DELETE SET NULL,
        FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE CASCADE
      );

      -- Task assignments table for multiple assignees
      CREATE TABLE IF NOT EXISTS task_assignments (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        task_id INTEGER NOT NULL,
        user_id INTEGER NOT NULL,
        assigned_by INTEGER NOT NULL,
        assigned_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (task_id) REFERENCES tasks(id) ON DELETE CASCADE,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        FOREIGN KEY (assigned_by) REFERENCES users(id) ON DELETE CASCADE,
        UNIQUE(task_id, user_id)
      );

      -- User permissions table for system-wide permissions
      CREATE TABLE IF NOT EXISTS user_permissions (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER NOT NULL,
        permission_name TEXT NOT NULL,
        permission_value BOOLEAN DEFAULT FALSE,
        granted_by INTEGER,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        FOREIGN KEY (granted_by) REFERENCES users(id) ON DELETE SET NULL,
        UNIQUE(user_id, permission_name)
      );

      -- Activity logs table
      CREATE TABLE IF NOT EXISTS activity_logs (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER NOT NULL,
        action TEXT NOT NULL,
        entity_type TEXT NOT NULL,
        entity_id INTEGER NOT NULL,
        details TEXT,
        ip_address TEXT,
        user_agent TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
      );

      CREATE INDEX IF NOT EXISTS idx_activity_logs_user ON activity_logs(user_id);
      CREATE INDEX IF NOT EXISTS idx_activity_logs_entity ON activity_logs(entity_type, entity_id);
      CREATE INDEX IF NOT EXISTS idx_activity_logs_created_at ON activity_logs(created_at);

      -- Sprints table
      CREATE TABLE IF NOT EXISTS sprints (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        project_id INTEGER NOT NULL,
        name TEXT NOT NULL,
        description TEXT,
        start_date DATE NOT NULL,
        end_date DATE NOT NULL,
        status TEXT DEFAULT 'planning' CHECK(status IN ('planning', 'active', 'completed', 'cancelled')),
        goal TEXT,
        created_by INTEGER NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE CASCADE,
        FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE CASCADE
      );
    `;

    db.exec(createTablesSQL, (err) => {
      if (err) {
        console.error('❌ Error creating tables:', err.message);
        reject(err);
      } else {
        console.log('✅ Database tables created/verified');
        seedInitialData().then(resolve).catch(reject);
      }
    });
  });
};

const seedInitialData = () => {
  return new Promise((resolve, reject) => {
    const seedSQL = `
      -- Normalize legacy roles to new simplified schema
      UPDATE users SET role = 'main_admin' WHERE role IN ('main_admin', 'super_admin');
      UPDATE users SET role = 'admin' WHERE role IN ('company_admin', 'manager', 'team_leader', 'sub_admin');
      UPDATE users SET role = 'user' WHERE role NOT IN ('main_admin', 'admin');
      UPDATE users SET permission_level = CASE role
        WHEN 'main_admin' THEN 3
        WHEN 'admin' THEN 2
        ELSE 1
      END WHERE permission_level NOT IN (1, 2, 3);

      UPDATE project_members SET role = 'main_admin' WHERE role IN ('main_admin', 'super_admin');
      UPDATE project_members SET role = 'admin' WHERE role IN ('manager', 'team_leader');
      UPDATE project_members SET role = 'user' WHERE role NOT IN ('main_admin', 'admin');

      UPDATE company_users SET role = 'admin' WHERE role IN ('admin', 'company_admin');
      UPDATE company_users SET role = 'user' WHERE role NOT IN ('admin');

      -- Insert main company
      INSERT OR IGNORE INTO companies (id, name, description, admin_phone)
      VALUES (1, 'Main Company', 'Main administrative company', '1234567890');

      -- Insert CompanyA
      INSERT OR IGNORE INTO companies (id, name, description, admin_phone)
      VALUES (2, 'CompanyA', 'Company A for testing', '1234512345');

      -- Insert main admin user
      INSERT OR IGNORE INTO users (id, username, phone_number, first_name, last_name, role, permission_level, company_id, status)
      VALUES (1, 'main_admin', '1234567890', 'Main', 'Admin', 'main_admin', 3, 1, 'active');

      -- Insert default admin user
      INSERT OR IGNORE INTO users (id, username, phone_number, first_name, last_name, role, permission_level, company_id, status, created_by)
      VALUES (2, 'primary_admin', '1234512345', 'Primary', 'Admin', 'admin', 2, 1, 'active', 1);

      -- Insert sample standard user
      INSERT OR IGNORE INTO users (id, username, phone_number, first_name, last_name, role, permission_level, company_id, status, created_by)
      VALUES (3, 'standard_user', '1234567899', 'Sample', 'User', 'user', 1, 1, 'active', 2);

      -- Link users to company
      INSERT OR IGNORE INTO company_users (company_id, user_id, role)
      VALUES 
        (1, 1, 'admin'),
        (1, 2, 'admin'),
        (1, 3, 'user');
    `;

    db.exec(seedSQL, (err) => {
      if (err) {
        console.error('❌ Error seeding initial data:', err.message);
        reject(err);
      } else {
        console.log('✅ Initial data seeded successfully');
        resolve();
      }
    });
  });
};

const executeQuery = (sql, params = []) => {
  return new Promise((resolve, reject) => {
    if (!db) {
      reject(new Error('Database not initialized'));
      return;
    }

    if (sql.trim().toUpperCase().startsWith('SELECT')) {
      db.all(sql, params, (err, rows) => {
        if (err) {
          reject(err);
        } else {
          resolve(rows);
        }
      });
    } else {
      db.run(sql, params, function(err) {
        if (err) {
          reject(err);
        } else {
          resolve({ insertId: this.lastID, changes: this.changes });
        }
      });
    }
  });
};

const closeDatabase = () => {
  return new Promise((resolve) => {
    if (db) {
      db.close((err) => {
        if (err) {
          console.error('❌ Error closing database:', err.message);
        } else {
          console.log('✅ Database connection closed');
        }
        resolve();
      });
    } else {
      resolve();
    }
  });
};

module.exports = {
  initDatabase,
  resetDatabase,
  executeQuery,
  closeDatabase
};
