const express = require('express');
const router = express.Router();
const authService = require('../services/authService');
const notificationService = require('../services/notificationService');
const { authenticateToken } = require('../middleware/auth');
const {
  validateLogin,
  validateOTPVerification,
  handleValidationErrors
} = require('../middleware/validation');

// Send OTP
router.post('/send-otp', validateLogin(), handleValidationErrors, async (req, res) => {
  try {
    const { phone_number } = req.body;
    
    const result = await authService.sendOTP(phone_number);
    
    res.status(200).json(result);
  } catch (error) {
    console.error('Send OTP error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to send OTP'
    });
  }
});

// Verify OTP and Login
router.post('/login', validateOTPVerification(), handleValidationErrors, async (req, res) => {
  try {
    const { phone_number, otp } = req.body;
    
    const result = await authService.login(phone_number, otp);
    
    if (result.success) {
      res.status(200).json(result);
    } else {
      res.status(401).json(result);
    }
  } catch (error) {
    console.error('Login error:', error);
    res.status(500).json({
      success: false,
      message: 'Login failed'
    });
  }
});

// Refresh Token
router.post('/refresh-token', async (req, res) => {
  try {
    const { refresh_token } = req.body;
    
    if (!refresh_token) {
      return res.status(400).json({
        success: false,
        message: 'Refresh token is required'
      });
    }
    
    const result = await authService.refreshToken(refresh_token);
    
    if (result.success) {
      res.status(200).json(result);
    } else {
      res.status(401).json(result);
    }
  } catch (error) {
    console.error('Refresh token error:', error);
    res.status(500).json({
      success: false,
      message: 'Token refresh failed'
    });
  }
});

// Logout
router.post('/logout', authenticateToken, async (req, res) => {
  try {
    const { refresh_token } = req.body;
    const userId = req.user.id;
    
    if (!refresh_token) {
      return res.status(400).json({
        success: false,
        message: 'Refresh token is required'
      });
    }
    
    const result = await authService.logout(userId, refresh_token);
    
    res.status(200).json(result);
  } catch (error) {
    console.error('Logout error:', error);
    res.status(500).json({
      success: false,
      message: 'Logout failed'
    });
  }
});

// Logout from all devices
router.post('/logout-all', authenticateToken, async (req, res) => {
  try {
    const userId = req.user.id;
    
    const result = await authService.logoutAll(userId);
    
    res.status(200).json(result);
  } catch (error) {
    console.error('Logout all error:', error);
    res.status(500).json({
      success: false,
      message: 'Logout from all devices failed'
    });
  }
});

// Get current user profile
router.get('/profile', authenticateToken, async (req, res) => {
  try {
    const user = req.user;
    
    res.status(200).json({
      success: true,
      message: 'Profile retrieved successfully',
      data: {
        user: {
          id: user.id,
          phone_number: user.phone_number,
          email: user.email,
          first_name: user.first_name,
          last_name: user.last_name,
          role: user.role,
          profile_image: user.profile_image
        }
      }
    });
  } catch (error) {
    console.error('Get profile error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to retrieve profile'
    });
  }
});

// Verify token (for frontend to check if token is valid)
router.get('/verify-token', authenticateToken, async (req, res) => {
  try {
    res.status(200).json({
      success: true,
      message: 'Token is valid',
      data: {
        user: {
          id: req.user.id,
          phone_number: req.user.phone_number,
          email: req.user.email,
          first_name: req.user.first_name,
          last_name: req.user.last_name,
          role: req.user.role
        }
      }
    });
  } catch (error) {
    console.error('Verify token error:', error);
    res.status(500).json({
      success: false,
      message: 'Token verification failed'
    });
  }
});

// Update FCM token
router.post('/update-fcm-token', authenticateToken, async (req, res) => {
  try {
    const { fcm_token } = req.body;
    const userId = req.user.id;
    
    if (!fcm_token) {
      return res.status(400).json({
        success: false,
        message: 'FCM token is required'
      });
    }
    
    const result = await notificationService.updateUserFcmToken(userId, fcm_token);
    
    if (result.success) {
      res.status(200).json({
        success: true,
        message: 'FCM token updated successfully'
      });
    } else {
      res.status(500).json({
        success: false,
        message: 'Failed to update FCM token'
      });
    }
  } catch (error) {
    console.error('Update FCM token error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update FCM token'
    });
  }
});

module.exports = router;