const express = require('express');
const router = express.Router();
const { executeQuery } = require('../config/database');
const { authenticateToken, requireMainAdmin, requireCompanyAdmin } = require('../middleware/auth');
const { validateId, validatePagination, handleValidationErrors } = require('../middleware/validation');

// Get all companies (Main Admin only)
router.get('/', authenticateToken, requireMainAdmin, validatePagination(), handleValidationErrors, async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const offset = (page - 1) * limit;
    const search = req.query.search || '';

    let whereClause = 'WHERE 1=1';
    let queryParams = [];

    if (search) {
      whereClause += ' AND (name LIKE ? OR description LIKE ?)';
      const searchPattern = `%${search}%`;
      queryParams.push(searchPattern, searchPattern);
    }

    // Get total count
    const countQuery = `
      SELECT COUNT(*) as total
      FROM companies
      ${whereClause}
    `;
    
    const countResult = await executeQuery(countQuery, queryParams);
    const total = countResult[0].total;

    // Get companies with pagination
    const query = `
      SELECT 
        c.id, 
        c.name, 
        c.description, 
        c.status,
        c.created_at, 
        c.updated_at,
        (SELECT COUNT(*) FROM users WHERE company_id = c.id) as user_count
      FROM companies c
      ${whereClause}
      ORDER BY c.created_at DESC
      LIMIT ${limit} OFFSET ${offset}
    `;
    
    const companies = await executeQuery(query, queryParams);

    res.json({
      data: companies,
      meta: {
        total,
        page,
        limit,
        last_page: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    console.error('Error fetching companies:', error);
    res.status(500).json({ message: 'Failed to fetch companies' });
  }
});

// Get company by ID
router.get('/:id', authenticateToken, requireCompanyAdmin, validateId('id'), handleValidationErrors, async (req, res) => {
  try {
    const companyId = req.params.id;
    
    // Company admins can only view their own company
    if (req.user.role === 'admin' && req.user.company_id !== parseInt(companyId)) {
      return res.status(403).json({ message: 'You do not have permission to view this company' });
    }

    const query = `
      SELECT 
        c.id, 
        c.name, 
        c.description, 
        c.status,
        c.created_at, 
        c.updated_at,
        (SELECT COUNT(*) FROM users WHERE company_id = c.id) as user_count
      FROM companies c
      WHERE c.id = ?
    `;
    
    const company = await executeQuery(query, [companyId]);

    if (company.length === 0) {
      return res.status(404).json({ message: 'Company not found' });
    }

    res.json(company[0]);
  } catch (error) {
    console.error('Error fetching company:', error);
    res.status(500).json({ message: 'Failed to fetch company' });
  }
});

// Create company (Main Admin only)
router.post('/', authenticateToken, requireMainAdmin, async (req, res) => {
  try {
    const { name, description, status = 'active' } = req.body;

    if (!name) {
      return res.status(400).json({ message: 'Company name is required' });
    }

    // Check if company with same name already exists
    const existingCompany = await executeQuery('SELECT id FROM companies WHERE name = ?', [name]);
    if (existingCompany.length > 0) {
      return res.status(400).json({ message: 'Company with this name already exists' });
    }

    const query = `
      INSERT INTO companies (name, description, status, created_at, updated_at)
      VALUES (?, ?, ?, NOW(), NOW())
    `;
    
    const result = await executeQuery(query, [name, description || '', status]);
    
    const newCompany = await executeQuery('SELECT * FROM companies WHERE id = ?', [result.lastID]);

    res.status(201).json(newCompany[0]);
  } catch (error) {
    console.error('Error creating company:', error);
    res.status(500).json({ message: 'Failed to create company' });
  }
});

// Update company
router.put('/:id', authenticateToken, requireCompanyAdmin, validateId('id'), handleValidationErrors, async (req, res) => {
  try {
    const companyId = req.params.id;
    const { name, description, status } = req.body;
    
    // Company admins can only update their own company and cannot change status
    if (req.user.role === 'admin') {
      if (req.user.company_id !== parseInt(companyId)) {
        return res.status(403).json({ message: 'You do not have permission to update this company' });
      }
      
      if (status) {
        return res.status(403).json({ message: 'Company admins cannot change company status' });
      }
    }

    // Check if company exists
    const existingCompany = await executeQuery('SELECT * FROM companies WHERE id = ?', [companyId]);
    if (existingCompany.length === 0) {
      return res.status(404).json({ message: 'Company not found' });
    }

    // Check if new name already exists for another company
    if (name && name !== existingCompany[0].name) {
      const nameExists = await executeQuery('SELECT id FROM companies WHERE name = ? AND id != ?', [name, companyId]);
      if (nameExists.length > 0) {
        return res.status(400).json({ message: 'Company with this name already exists' });
      }
    }

    // Build update query dynamically
    let updateFields = [];
    let queryParams = [];

    if (name) {
      updateFields.push('name = ?');
      queryParams.push(name);
    }

    if (description !== undefined) {
      updateFields.push('description = ?');
      queryParams.push(description);
    }

    if (status && req.user.role === 'main_admin') {
      updateFields.push('status = ?');
      queryParams.push(status);
    }

    if (updateFields.length === 0) {
      return res.status(400).json({ message: 'No fields to update' });
    }

    updateFields.push('updated_at = NOW()');
    
    const query = `
      UPDATE companies
      SET ${updateFields.join(', ')}
      WHERE id = ?
    `;
    
    queryParams.push(companyId);
    await executeQuery(query, queryParams);
    
    const updatedCompany = await executeQuery('SELECT * FROM companies WHERE id = ?', [companyId]);

    res.json(updatedCompany[0]);
  } catch (error) {
    console.error('Error updating company:', error);
    res.status(500).json({ message: 'Failed to update company' });
  }
});

// Delete company (Main Admin only)
router.delete('/:id', authenticateToken, requireMainAdmin, validateId('id'), handleValidationErrors, async (req, res) => {
  try {
    const companyId = req.params.id;

    // Check if company exists
    const existingCompany = await executeQuery('SELECT * FROM companies WHERE id = ?', [companyId]);
    if (existingCompany.length === 0) {
      return res.status(404).json({ message: 'Company not found' });
    }

    // Check if company has users
    const userCount = await executeQuery('SELECT COUNT(*) as count FROM users WHERE company_id = ?', [companyId]);
    if (userCount[0].count > 0) {
      return res.status(400).json({ 
        message: 'Cannot delete company with active users. Transfer or delete users first.',
        user_count: userCount[0].count
      });
    }

    // Delete company
    await executeQuery('DELETE FROM companies WHERE id = ?', [companyId]);

    res.json({ message: 'Company deleted successfully' });
  } catch (error) {
    console.error('Error deleting company:', error);
    res.status(500).json({ message: 'Failed to delete company' });
  }
});

// Get company statistics (Main Admin only)
router.get('/stats/overview', authenticateToken, requireMainAdmin, async (req, res) => {
  try {
    const stats = {
      total_companies: 0,
      active_companies: 0,
      inactive_companies: 0,
      new_companies_last_30_days: 0
    };

    // Get total companies
    const totalQuery = 'SELECT COUNT(*) as count FROM companies';
    const totalResult = await executeQuery(totalQuery);
    stats.total_companies = totalResult[0].count;

    // Get active companies
    const activeQuery = 'SELECT COUNT(*) as count FROM companies WHERE status = ?';
    const activeResult = await executeQuery(activeQuery, ['active']);
    stats.active_companies = activeResult[0].count;

    // Get inactive companies
    const inactiveResult = await executeQuery(activeQuery, ['inactive']);
    stats.inactive_companies = inactiveResult[0].count;

    // Get new companies in last 30 days
    const newCompaniesQuery = `
      SELECT COUNT(*) as count 
      FROM companies 
      WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
    `;
    const newCompaniesResult = await executeQuery(newCompaniesQuery);
    stats.new_companies_last_30_days = newCompaniesResult[0].count;

    res.json(stats);
  } catch (error) {
    console.error('Error fetching company statistics:', error);
    res.status(500).json({ message: 'Failed to fetch company statistics' });
  }
});

module.exports = router;
