const express = require('express');
const router = express.Router();
const { executeQuery } = require('../config/database');
const { authenticateToken } = require('../middleware/auth');
const iap = require('in-app-purchase');

// Configure IAP
iap.config({
  androidPublicKey: process.env.GOOGLE_PLAY_PUBLIC_KEY,
  applePassword: process.env.APPLE_SHARED_SECRET,
  googleAccToken: process.env.GOOGLE_API_ACCESS_TOKEN,
  googleRefToken: process.env.GOOGLE_API_REFRESH_TOKEN,
  googleClientID: process.env.GOOGLE_API_CLIENT_ID,
  googleClientSecret: process.env.GOOGLE_API_CLIENT_SECRET,
  verbose: true
});

// Initialize IAP
iap.setup();

// Get user's subscription status
router.get('/status', authenticateToken, async (req, res) => {
  try {
    const userId = req.user.id;
    
    const subscriptions = await executeQuery(
      `SELECT * FROM subscriptions 
       WHERE user_id = ? 
       ORDER BY created_at DESC 
       LIMIT 1`,
      [userId]
    );
    
    if (subscriptions.length === 0) {
      return res.status(200).json({
        success: true,
        message: 'No active subscription found',
        data: {
          subscription: null
        }
      });
    }
    
    const subscription = subscriptions[0];
    
    res.status(200).json({
      success: true,
      message: 'Subscription status retrieved successfully',
      data: {
        subscription
      }
    });
  } catch (error) {
    console.error('Get subscription status error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to retrieve subscription status'
    });
  }
});

// Verify subscription purchase (called by mobile app after purchase)
router.post('/verify', authenticateToken, async (req, res) => {
  try {
    const { platform, receipt_data, product_id } = req.body;
    const userId = req.user.id;
    
    // Validate input
    if (!platform || !receipt_data || !product_id) {
      return res.status(400).json({
        success: false,
        message: 'Platform, receipt_data, and product_id are required'
      });
    }
    
    // Verify the purchase with the respective store
    let verificationResult;
    
    if (platform === 'ios') {
      verificationResult = await verifyIOSPurchase(receipt_data, product_id);
    } else if (platform === 'android') {
      verificationResult = await verifyAndroidPurchase(receipt_data, product_id);
    } else {
      return res.status(400).json({
        success: false,
        message: 'Invalid platform specified'
      });
    }
    
    if (!verificationResult.success) {
      return res.status(400).json({
        success: false,
        message: 'Purchase verification failed',
        data: {
          error: verificationResult.error
        }
      });
    }
    
    // Create or update subscription record
    const subscriptionData = {
      user_id: userId,
      platform: platform,
      product_id: product_id,
      transaction_id: verificationResult.transaction_id,
      receipt_data: receipt_data,
      status: 'active',
      start_date: new Date().toISOString(),
      end_date: calculateEndDate(product_id),
      created_at: new Date().toISOString(),
      updated_at: new Date().toISOString()
    };
    
    // Check if subscription already exists
    const existingSubscriptions = await executeQuery(
      'SELECT id FROM subscriptions WHERE transaction_id = ?',
      [verificationResult.transaction_id]
    );
    
    let subscriptionId;
    
    if (existingSubscriptions.length > 0) {
      // Update existing subscription
      subscriptionId = existingSubscriptions[0].id;
      await executeQuery(
        `UPDATE subscriptions 
         SET status = ?, end_date = ?, updated_at = ? 
         WHERE id = ?`,
        [subscriptionData.status, subscriptionData.end_date, subscriptionData.updated_at, subscriptionId]
      );
    } else {
      // Create new subscription
      const result = await executeQuery(
        `INSERT INTO subscriptions 
         (user_id, platform, product_id, transaction_id, receipt_data, status, start_date, end_date, created_at, updated_at) 
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [
          subscriptionData.user_id,
          subscriptionData.platform,
          subscriptionData.product_id,
          subscriptionResult.transactionId,
          subscriptionData.receipt_data,
          subscriptionData.status,
          subscriptionData.start_date,
          subscriptionData.end_date,
          subscriptionData.created_at,
          subscriptionData.updated_at
        ]
      );
      subscriptionId = result.insertId;
    }
    
    // Update user's subscription status
    await updateUserSubscriptionStatus(userId, subscriptionData.status, subscriptionData.end_date);
    
    // Get the updated subscription
    const subscriptions = await executeQuery(
      'SELECT * FROM subscriptions WHERE id = ?',
      [subscriptionId]
    );
    
    res.status(200).json({
      success: true,
      message: 'Subscription verified and activated successfully',
      data: {
        subscription: subscriptions[0]
      }
    });
  } catch (error) {
    console.error('Subscription verification error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to verify subscription'
    });
  }
});

// Webhook endpoint for iOS App Store notifications
router.post('/webhook/ios', async (req, res) => {
  try {
    const notification = req.body;
    
    // Process the App Store notification
    await processIOSNotification(notification);
    
    res.status(200).json({
      success: true,
      message: 'Notification processed successfully'
    });
  } catch (error) {
    console.error('iOS webhook error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to process notification'
    });
  }
});

// Webhook endpoint for Google Play notifications
router.post('/webhook/android', async (req, res) => {
  try {
    const notification = req.body;
    
    // Process the Google Play notification
    await processAndroidNotification(notification);
    
    res.status(200).json({
      success: true,
      message: 'Notification processed successfully'
    });
  } catch (error) {
    console.error('Android webhook error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to process notification'
    });
  }
});

// Helper functions
async function verifyIOSPurchase(receiptData, productId) {
  try {
    // For iOS, we need to validate the receipt with Apple's servers
    const validationData = {
      receipt: receiptData,
      secret: process.env.APPLE_SHARED_SECRET
    };
    
    const result = await iap.validate(validationData);
    
    if (result.status === 0) {
      // Valid receipt
      const latestReceiptInfo = result.latest_receipt_info || [];
      const transaction = latestReceiptInfo.find(item => item.product_id === productId);
      
      if (transaction) {
        return {
          success: true,
          transaction_id: transaction.transaction_id,
          product_id: transaction.product_id,
          expires_date: transaction.expires_date
        };
      }
    }
    
    return {
      success: false,
      error: 'Invalid receipt or product not found'
    };
  } catch (error) {
    console.error('iOS verification error:', error);
    return {
      success: false,
      error: error.message
    };
  }
}

async function verifyAndroidPurchase(receiptData, productId) {
  try {
    // For Android, we need to validate the purchase with Google's servers
    // receiptData should contain the purchase token
    const result = await iap.validateGooglePlayReceipt({
      data: receiptData,
      signature: '' // If needed
    });
    
    if (result.isSubscription === true && result.validated === true) {
      return {
        success: true,
        transaction_id: result.orderId,
        product_id: result.productId,
        expires_date: result.expirationTime
      };
    }
    
    return {
      success: false,
      error: 'Invalid purchase or product not found'
    };
  } catch (error) {
    console.error('Android verification error:', error);
    return {
      success: false,
      error: error.message
    };
  }
}

function calculateEndDate(productId) {
  // Calculate end date based on product ID
  // This is a simplified example - in reality, you should use the expiration date from the store
  const now = new Date();
  now.setMonth(now.getMonth() + 1); // Add one month
  return now.toISOString();
}

async function updateUserSubscriptionStatus(userId, status, endDate) {
  // Update user's subscription status in the users table
  await executeQuery(
    `UPDATE users 
     SET subscription_status = ?, subscription_end_date = ?, updated_at = ? 
     WHERE id = ?`,
    [status, endDate, new Date().toISOString(), userId]
  );
}

async function processIOSNotification(notification) {
  // Process iOS App Store notification
  console.log('Processing iOS notification:', notification);
  
  // Handle different notification types (renewal, cancellation, etc.)
  // Update subscription status accordingly
  try {
    // Parse the notification
    const decodedPayload = JSON.parse(notification);
    
    // Handle different notification types
    switch (decodedPayload.notification_type) {
      case 'DID_RENEW':
        // Subscription renewed
        await updateSubscriptionStatus(decodedPayload.transaction_id, 'active');
        break;
      case 'CANCEL':
        // Subscription cancelled
        await updateSubscriptionStatus(decodedPayload.transaction_id, 'cancelled');
        break;
      case 'EXPIRE':
        // Subscription expired
        await updateSubscriptionStatus(decodedPayload.transaction_id, 'expired');
        break;
      default:
        console.log('Unhandled notification type:', decodedPayload.notification_type);
    }
  } catch (error) {
    console.error('Error processing iOS notification:', error);
  }
}

async function processAndroidNotification(notification) {
  // Process Google Play notification
  console.log('Processing Android notification:', notification);
  
  // Handle different notification types (renewal, cancellation, etc.)
  // Update subscription status accordingly
  try {
    // Parse the notification
    const decodedPayload = JSON.parse(notification);
    
    // Handle different notification types
    if (decodedPayload.subscriptionNotification) {
      const notificationType = decodedPayload.subscriptionNotification.notificationType;
      
      switch (notificationType) {
        case 2: // SUBSCRIPTION_RENEWED
          // Subscription renewed
          await updateSubscriptionStatus(decodedPayload.subscriptionNotification.purchaseToken, 'active');
          break;
        case 3: // SUBSCRIPTION_CANCELED
          // Subscription cancelled
          await updateSubscriptionStatus(decodedPayload.subscriptionNotification.purchaseToken, 'cancelled');
          break;
        case 12: // SUBSCRIPTION_EXPIRED
          // Subscription expired
          await updateSubscriptionStatus(decodedPayload.subscriptionNotification.purchaseToken, 'expired');
          break;
        default:
          console.log('Unhandled notification type:', notificationType);
      }
    }
  } catch (error) {
    console.error('Error processing Android notification:', error);
  }
}

async function updateSubscriptionStatus(transactionId, status) {
  try {
    // Update subscription status
    await executeQuery(
      `UPDATE subscriptions 
       SET status = ?, updated_at = ? 
       WHERE transaction_id = ?`,
      [status, new Date().toISOString(), transactionId]
    );
    
    // Also update the user's subscription status
    await executeQuery(
      `UPDATE users u
       JOIN subscriptions s ON u.id = s.user_id
       SET u.subscription_status = ?, u.subscription_end_date = s.end_date, u.updated_at = ?
       WHERE s.transaction_id = ?`,
      [status, new Date().toISOString(), transactionId]
    );
  } catch (error) {
    console.error('Error updating subscription status:', error);
  }
}

module.exports = router;