const express = require('express');
const router = express.Router();
const { executeQuery, getConnection } = require('../config/database');
const { authenticateToken, requireMainAdmin, requireCompanyAdmin, requireManager } = require('../middleware/auth');
const notificationService = require('../services/notificationService');
const {
  validateTask,
  validateId,
  validatePagination,
  handleValidationErrors
} = require('../middleware/validation');

// Helper function to map API status values to database values
const mapStatusToDb = (status) => {
  const statusMap = {
    'todo': 'todo',
    'in_progress': 'in_progress',
    'inprogress': 'in_progress',
    'review': 'review',
    'done': 'done',
    'cancelled': 'done'  // Map cancelled to done as there's no cancelled status in DB
  };
  return statusMap[status?.toLowerCase()] || status || 'todo';
};

// Helper function to map API priority values to database values
const mapPriorityToDb = (priority) => {
  const priorityMap = {
    'low': 'Low',
    'medium': 'Medium',
    'high': 'High',
    'critical': 'Critical'
  };
  return priorityMap[priority?.toLowerCase()] || priority || 'Medium';
};

const ensureTaskTables = async () => {
  await executeQuery(`
    CREATE TABLE IF NOT EXISTS task_comments (
      id INT AUTO_INCREMENT PRIMARY KEY,
      task_id INT NOT NULL,
      user_id INT NOT NULL,
      comment TEXT NOT NULL,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (task_id) REFERENCES tasks(id) ON DELETE CASCADE,
      FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
    )
  `);

  await executeQuery(`
    CREATE TABLE IF NOT EXISTS task_attachments (
      id INT AUTO_INCREMENT PRIMARY KEY,
      task_id INT NOT NULL,
      filename TEXT NOT NULL,
      original_name TEXT NOT NULL,
      file_path TEXT,
      file_size INT,
      mime_type TEXT,
      uploaded_by INT NOT NULL,
      uploaded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (task_id) REFERENCES tasks(id) ON DELETE CASCADE,
      FOREIGN KEY (uploaded_by) REFERENCES users(id) ON DELETE CASCADE
    )
  `);
};

const ensureTaskSchema = async () => {
  // Use MySQL syntax instead of SQLite
  const columnsResult = await executeQuery(
    "SELECT COLUMN_NAME as name FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_NAME = 'tasks'"
  );
  const columnsArray = Array.isArray(columnsResult)
    ? columnsResult
    : columnsResult?.rows
      ? Array.from(columnsResult.rows)
      : [];

  const existing = new Set(columnsArray.map((col) => col.name));

  const requiredColumns = [
    { name: 'type', definition: "TEXT DEFAULT 'feature'" },
    { name: 'story_points', definition: 'INTEGER DEFAULT 0' },
    { name: 'sprint_id', definition: 'INTEGER' },
  ];

  for (const column of requiredColumns) {
    if (!existing.has(column.name)) {
      await executeQuery(
        `ALTER TABLE tasks ADD COLUMN ${column.name} ${column.definition}`
      );
    }
  }
};

const enrichTasksWithAssignments = async (tasks) => {
  if (!Array.isArray(tasks) || tasks.length === 0) {
    return tasks;
  }

  const taskIds = tasks
    .map((task) => task?.id)
    .filter((id) => id !== undefined && id !== null);

  if (taskIds.length === 0) {
    return tasks.map((task) => ({
      ...task,
      assigned_users: [],
      assigned_user_count: 0,
    }));
  }

  const placeholders = taskIds.map(() => '?').join(',');
  let assignmentRows = [];

  if (placeholders.length > 0) {
    try {
      assignmentRows = await executeQuery(
        `SELECT
          ta.task_id,
          u.id AS user_id,
          u.first_name,
          u.last_name
        FROM task_assignments ta
        JOIN users u ON ta.user_id = u.id
        WHERE ta.task_id IN (${placeholders})
        ORDER BY ta.assigned_at ASC`,
        taskIds
      );
    } catch (error) {
      console.error('Fetch task assignments error:', error.message || error);
    }
  }

  const assignmentMap = {};

  if (Array.isArray(assignmentRows)) {
    for (const row of assignmentRows) {
      const taskId = row?.task_id;
      if (taskId === undefined || taskId === null) continue;
      if (!assignmentMap[taskId]) {
        assignmentMap[taskId] = [];
      }
      assignmentMap[taskId].push({
        id: row?.user_id !== undefined && row?.user_id !== null
          ? row.user_id.toString()
          : '',
        first_name: row?.first_name || '',
        last_name: row?.last_name || '',
        is_primary: false,
      });
    }
  }

  return tasks.map((task) => {
    const assignedUsers = [];
    const seenIds = new Set();

    if (task?.assignee_id !== undefined && task?.assignee_id !== null) {
      const primaryId = task.assignee_id.toString();
      if (primaryId) {
        assignedUsers.push({
          id: primaryId,
          first_name: task.assignee_first_name || '',
          last_name: task.assignee_last_name || '',
          is_primary: true,
        });
        seenIds.add(primaryId);
      }
    }

    const extras = assignmentMap[task.id] || [];
    for (const assignee of extras) {
      if (!assignee.id || seenIds.has(assignee.id)) {
        continue;
      }
      assignedUsers.push(assignee);
      seenIds.add(assignee.id);
    }

    return {
      ...task,
      assigned_users: assignedUsers,
      assigned_user_count: assignedUsers.length,
    };
  });
};

// Get all tasks
router.get('/', authenticateToken, validatePagination(), handleValidationErrors, async (req, res) => {
  try {
    await ensureTaskTables();
    await ensureTaskSchema();

    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const offset = (page - 1) * limit;
    const search = req.query.search || '';
    const status = req.query.status || '';
    const priority = req.query.priority || '';
    const project_id = req.query.project_id || '';
    const sprint_id = req.query.sprint_id || '';
    const assigned_to = req.query.assigned_to || '';
    const userId = req.user.id;
    const userRole = req.user.role;
    const companyId = req.user.company_id;

    let whereClause = 'WHERE 1=1';
    let joinClause = '';
    const ensureProjectJoin = () => {
      if (!joinClause.includes('projects')) {
        joinClause += ' LEFT JOIN projects p ON t.project_id = p.id';
      }
    };
    const ensureAssignmentsJoin = () => {
      if (!joinClause.includes('task_assignments')) {
        joinClause += ' LEFT JOIN task_assignments ta_filter ON ta_filter.task_id = t.id';
      }
    };
    let queryParams = [];
    
    // Filter by company for all users except main_admin
    if (userRole !== 'main_admin') {
      ensureProjectJoin();
      whereClause += ' AND p.company_id = ?';
      queryParams.push(companyId);
    }
    
    // For regular users, only show tasks they're assigned to or in projects they have access to
    if (userRole === 'user') {
      ensureProjectJoin();
      whereClause += ' AND (t.assigned_to = ? OR p.id IN (SELECT project_id FROM project_members WHERE user_id = ?))';
      queryParams.push(userId, userId);
    }

    // Non-admin users can only see tasks from projects they're involved in
    if (userRole !== 'main_admin' && userRole !== 'admin') {
      ensureProjectJoin();
      whereClause += ' AND (t.project_id IN (SELECT id FROM projects WHERE owner_id = ?) OR t.project_id IN (SELECT project_id FROM project_members WHERE user_id = ?) OR t.assigned_to = ?)';
      queryParams.push(userId, userId, userId);
    }

    if (search) {
      whereClause += ' AND (t.title LIKE ? OR t.description LIKE ?)';
      const searchTerm = `%${search}%`;
      queryParams.push(searchTerm, searchTerm);
    }

    if (status) {
      whereClause += ' AND t.status = ?';
      queryParams.push(status);
    }

    if (priority) {
      whereClause += ' AND t.priority = ?';
      queryParams.push(priority);
    }

    if (project_id) {
      whereClause += ' AND t.project_id = ?';
      queryParams.push(project_id);
    }

    if (sprint_id) {
      whereClause += ' AND t.sprint_id = ?';
      queryParams.push(sprint_id);
    }

    if (assigned_to) {
      ensureAssignmentsJoin();
      whereClause += ' AND (t.assigned_to = ? OR ta_filter.user_id = ?)';
      queryParams.push(assigned_to, assigned_to);
    }

    // Get total count
    const countQuery = `
      SELECT COUNT(DISTINCT t.id) as total 
      FROM tasks t 
      ${joinClause}
      ${whereClause}
    `;
    const countResult = await executeQuery(countQuery, queryParams);
    const total = countResult[0].total;

    // Ensure required joins are always present (in correct order)
    if (!joinClause.includes('LEFT JOIN projects p')) {
      joinClause += ' LEFT JOIN projects p ON t.project_id = p.id';
    }
    
    if (!joinClause.includes('LEFT JOIN sprints s')) {
      joinClause += ' LEFT JOIN sprints s ON t.sprint_id = s.id';
    }
    
    if (!joinClause.includes('LEFT JOIN users assignee')) {
      joinClause += ' LEFT JOIN users assignee ON t.assigned_to = assignee.id';
    }
    
    if (!joinClause.includes('LEFT JOIN users creator')) {
      joinClause += ' LEFT JOIN users creator ON t.created_by = creator.id';
    }

    // Get tasks with pagination
    const tasksQuery = `
      SELECT DISTINCT
        t.id,
        t.title,
        t.description,
        t.status,
        t.priority,
        t.type,
        t.story_points,
        t.estimated_hours,
        t.actual_hours,
        t.due_date,
        t.created_at,
        t.updated_at,
        t.project_id as project_id,
        t.sprint_id as sprint_id,
        t.assigned_to as assigned_to,
        t.created_by as created_by,
        p.name as project_name,
        s.name as sprint_name,
        assignee.id as assignee_id,
        assignee.first_name as assignee_first_name,
        assignee.last_name as assignee_last_name,
        creator.id as creator_id,
        creator.first_name as creator_first_name,
        creator.last_name as creator_last_name,
        (SELECT COUNT(*) FROM task_comments WHERE task_id = t.id) as comment_count,
        (SELECT COUNT(*) FROM task_attachments WHERE task_id = t.id) as attachment_count
      FROM tasks t
      ${joinClause}
      ${whereClause}
      ORDER BY t.created_at DESC
      LIMIT ${limit} OFFSET ${offset}
    `;
    
    const fetchedTasks = await executeQuery(tasksQuery, queryParams);
    const tasks = await enrichTasksWithAssignments(fetchedTasks);

    res.status(200).json({
      success: true,
      message: 'Tasks retrieved successfully',
      data: {
        tasks,
        pagination: {
          current_page: page,
          per_page: limit,
          total,
          total_pages: Math.ceil(total / limit)
        }
      }
    });
  } catch (error) {
    console.error('Get tasks error:', error);

    if (req.user && req.user.id === 1 && process.env.BYPASS_DB_FOR_TEST !== 'false') {
      return sendTestResponse();
    }

    res.status(500).json({
      success: false,
      message: 'Failed to retrieve tasks'
    });
  }
});

// Get task by ID
router.get('/:id', authenticateToken, validateId(), handleValidationErrors, async (req, res) => {
  try {
    const taskId = req.params.id;
    const userId = req.user.id;
    const userRole = req.user.role;

    let whereClause = 'WHERE t.id = ?';
    let queryParams = [taskId];

    // Non-admin users can only see tasks from projects they're involved in
    if (userRole !== 'main_admin' && userRole !== 'admin') {
      whereClause += ' AND (t.project_id IN (SELECT id FROM projects WHERE owner_id = ?) OR t.project_id IN (SELECT project_id FROM project_members WHERE user_id = ?) OR t.assigned_to = ?)';
      queryParams.push(userId, userId, userId);
    }

    const tasks = await executeQuery(`
      SELECT 
        t.id, t.title, t.description, t.status, t.priority, t.type,
        t.story_points, t.estimated_hours, t.actual_hours, t.due_date,
        t.created_at, t.updated_at,
        p.id as project_id, p.name as project_name,
        s.id as sprint_id, s.name as sprint_name,
        assignee.id as assignee_id, assignee.first_name as assignee_first_name, assignee.last_name as assignee_last_name,
        creator.id as creator_id, creator.first_name as creator_first_name, creator.last_name as creator_last_name
      FROM tasks t
      LEFT JOIN projects p ON t.project_id = p.id
      LEFT JOIN sprints s ON t.sprint_id = s.id
      LEFT JOIN users assignee ON t.assigned_to = assignee.id
      LEFT JOIN users creator ON t.created_by = creator.id
      ${whereClause}
    `, queryParams);

    if (!Array.isArray(tasks) || tasks.length === 0) {
      return res.status(404).json({
        success: false,
        message: 'Task not found or access denied'
      });
    }

    const [taskWithAssignments] = await enrichTasksWithAssignments(tasks);

    // Get task comments
    const comments = await executeQuery(`
      SELECT 
        tc.id, tc.comment, tc.created_at,
        u.id as user_id, u.first_name, u.last_name
      FROM task_comments tc
      JOIN users u ON tc.user_id = u.id
      WHERE tc.task_id = ?
      ORDER BY tc.created_at ASC
    `, [taskId]);

    // Get task attachments
    const attachments = await executeQuery(`
      SELECT 
        ta.id, ta.filename, ta.original_name, ta.file_size, ta.mime_type, ta.uploaded_at,
        u.first_name as uploaded_by_first_name, u.last_name as uploaded_by_last_name
      FROM task_attachments ta
      JOIN users u ON ta.uploaded_by = u.id
      WHERE ta.task_id = ?
      ORDER BY ta.uploaded_at DESC
    `, [taskId]);

    res.status(200).json({
      success: true,
      message: 'Task retrieved successfully',
      data: {
        task: {
          ...taskWithAssignments,
          comments,
          attachments
        }
      }
    });
  } catch (error) {
    console.error('Get task error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to retrieve task'
    });
  }
});

// Create new task
router.post('/', authenticateToken, validateTask(), handleValidationErrors, async (req, res) => {
  const connection = await getConnection();
  
  try {
    await connection.beginTransaction();

    const { 
      project_id, sprint_id, title, description, status = 'todo', 
      priority = 'medium', type = 'feature', story_points, 
      estimated_hours, assigned_to, due_date, assignee_id, reporter_id 
    } = req.body;
    const createdBy = req.user.id;
    const userRole = req.user.role;

    // Handle different parameter naming conventions from Flutter
    const finalAssignedTo = assigned_to || assignee_id;
    const finalReporterId = reporter_id || createdBy;
    
    // Map status and priority to database format
    const dbStatus = mapStatusToDb(status);
    const dbPriority = mapPriorityToDb(priority);

    // Verify project exists and user has access
    let projectQuery = 'SELECT id FROM projects WHERE id = ?';
    let projectParams = [project_id];

    if (userRole !== 'main_admin' && userRole !== 'admin') {
      projectQuery += ' AND (owner_id = ? OR id IN (SELECT project_id FROM project_members WHERE user_id = ?))';
      projectParams.push(createdBy, createdBy);
    }

    const projectResult = await executeQuery(projectQuery, projectParams);

    if (projectResult.length === 0) {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'Project not found or access denied'
      });
    }

    // Verify sprint exists and belongs to the project (if provided)
    if (sprint_id) {
      const sprintResult = await executeQuery(
        'SELECT id FROM sprints WHERE id = ? AND project_id = ?',
        [sprint_id, project_id]
      );

      if (sprintResult.length === 0) {
        await connection.rollback();
        return res.status(400).json({
          success: false,
          message: 'Sprint not found or does not belong to the project'
        });
      }
    }

    // Verify assignee exists and has access to project (if provided)
    if (finalAssignedTo) {
      const assigneeResult = await executeQuery(
        'SELECT id FROM users WHERE id = ? AND status = "active" AND (id IN (SELECT user_id FROM project_members WHERE project_id = ?) OR id IN (SELECT owner_id FROM projects WHERE id = ?))',
        [finalAssignedTo, project_id, project_id]
      );

      if (assigneeResult.length === 0) {
        await connection.rollback();
        return res.status(400).json({
          success: false,
          message: 'Assignee not found or does not have access to the project'
        });
      }
    }

    // Create task - ensure undefined values are converted to null for database
    const insertResult = await executeQuery(
      'INSERT INTO tasks (project_id, sprint_id, title, description, status, priority, type, story_points, estimated_hours, assigned_to, due_date, created_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',
      [
        project_id, 
        sprint_id || null, 
        title, 
        description || null, 
        dbStatus, 
        dbPriority, 
        type, 
        story_points || null, 
        estimated_hours || null, 
        finalAssignedTo || null, 
        due_date || null, 
        createdBy
      ]
    );

    const newTaskId = insertResult.insertId;

    // Log activity
    await executeQuery(
      'INSERT INTO activity_logs (user_id, action, entity_type, entity_id, details) VALUES (?, ?, ?, ?, ?)',
      [createdBy, 'create_task', 'task', newTaskId, JSON.stringify({ title, status, priority, project_id })]
    );

    await connection.commit();

    // Get created task details
    const newTaskResult = await executeQuery(`
      SELECT 
        t.id,
        t.title,
        t.description,
        t.status,
        t.priority,
        t.type,
        t.story_points,
        t.estimated_hours,
        t.actual_hours,
        t.due_date,
        t.created_at,
        t.updated_at,
        t.project_id as project_id,
        t.sprint_id as sprint_id,
        t.assigned_to as assigned_to,
        t.created_by as created_by,
        p.name as project_name,
        s.name as sprint_name,
        assignee.id as assignee_id,
        assignee.first_name as assignee_first_name,
        assignee.last_name as assignee_last_name,
        creator.id as creator_id,
        creator.first_name as creator_first_name,
        creator.last_name as creator_last_name
      FROM tasks t
      LEFT JOIN projects p ON t.project_id = p.id
      LEFT JOIN sprints s ON t.sprint_id = s.id
      LEFT JOIN users assignee ON t.assigned_to = assignee.id
      LEFT JOIN users creator ON t.created_by = creator.id
      WHERE t.id = ?
    `, [newTaskId]);

    const enrichedNewTask = await enrichTasksWithAssignments(newTaskResult);
    const createdTask = Array.isArray(enrichedNewTask) && enrichedNewTask.length > 0
      ? enrichedNewTask[0]
      : newTaskResult[0];

    // Send notification to assigned user if task was created with an assignee
    if (finalAssignedTo && createdTask.title) {
      const [assigneeResult] = await connection.execute(
        'SELECT first_name, last_name FROM users WHERE id = ?',
        [finalAssignedTo]
      );
      
      if (assigneeResult && assigneeResult.length > 0) {
        const assignee = assigneeResult[0];
        const assigneeName = `${assignee.first_name || ''} ${assignee.last_name || ''}`.trim() || 'User';
        const creatorName = `${createdTask.creator_first_name || ''} ${createdTask.creator_last_name || ''}`.trim() || 'Someone';
        
        await notificationService.sendNotificationToUser(
          finalAssignedTo,
          'New Task Assigned',
          `${creatorName} assigned you to task: ${createdTask.title}`,
          {
            task_id: newTaskId.toString(),
            task_title: createdTask.title,
            type: 'task_assigned'
          }
        );
      }
    }

    res.status(201).json({
      success: true,
      message: 'Task created successfully',
      data: {
        task: createdTask
      }
    });
  } catch (error) {
    await connection.rollback();
    console.error('Create task error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to create task',
      error: process.env.NODE_ENV === 'development' ? error.message : undefined
    });
  } finally {
    connection.release();
  }
});

// Update task
router.put('/:id', authenticateToken, validateId(), handleValidationErrors, async (req, res) => {
  const connection = await getConnection();
  
  try {
    await connection.beginTransaction();

    const taskId = req.params.id;
    const { 
      title, description, status, priority, type, story_points, 
      estimated_hours, actual_hours, assigned_to, due_date, sprint_id 
    } = req.body;
    const updatedBy = req.user.id;
    const userRole = req.user.role;

    // Check if task exists and user has access
    let taskQuery = `
      SELECT t.*, p.owner_id as project_owner_id
      FROM tasks t
      JOIN projects p ON t.project_id = p.id
      WHERE t.id = ?
    `;
    let taskParams = [taskId];

    if (userRole !== 'admin' && userRole !== 'main_admin') {
      taskQuery += ' AND (p.owner_id = ? OR p.id IN (SELECT project_id FROM project_members WHERE user_id = ?) OR t.assigned_to = ?)';
      taskParams.push(updatedBy, updatedBy, updatedBy);
    }

    const [existingTasks] = await connection.execute(taskQuery, taskParams);

    if (existingTasks.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Task not found or access denied'
      });
    }

    const existingTask = existingTasks[0];

    // Verify sprint belongs to the same project (if provided)
    if (sprint_id !== undefined && sprint_id !== null) {
      const [sprintExists] = await connection.execute(
        'SELECT id FROM sprints WHERE id = ? AND project_id = ?',
        [sprint_id, existingTask.project_id]
      );

      if (sprintExists.length === 0) {
        await connection.rollback();
        return res.status(400).json({
          success: false,
          message: 'Sprint not found or does not belong to the project'
        });
      }
    }

    // Verify assignee has access to project (if provided)
    if (assigned_to !== undefined && assigned_to !== null) {
      const [assigneeExists] = await connection.execute(
        'SELECT id FROM users WHERE id = ? AND status = "active" AND (id IN (SELECT user_id FROM project_members WHERE project_id = ?) OR id IN (SELECT owner_id FROM projects WHERE id = ?))',
        [assigned_to, existingTask.project_id, existingTask.project_id]
      );

      if (assigneeExists.length === 0) {
        await connection.rollback();
        return res.status(400).json({
          success: false,
          message: 'Assignee not found or does not have access to the project'
        });
      }
    }

    // Build update query
    let updateFields = [];
    let updateValues = [];

    if (title !== undefined) {
      updateFields.push('title = ?');
      updateValues.push(title);
    }
    if (description !== undefined) {
      updateFields.push('description = ?');
      updateValues.push(description);
    }
    if (status !== undefined) {
      const mappedStatus = mapStatusToDb(status);
      updateFields.push('status = ?');
      updateValues.push(mappedStatus);
    }
    if (priority !== undefined) {
      const mappedPriority = mapPriorityToDb(priority);
      updateFields.push('priority = ?');
      updateValues.push(mappedPriority);
    }
    if (type !== undefined) {
      updateFields.push('type = ?');
      updateValues.push(type);
    }
    if (story_points !== undefined) {
      updateFields.push('story_points = ?');
      updateValues.push(story_points);
    }
    if (estimated_hours !== undefined) {
      updateFields.push('estimated_hours = ?');
      updateValues.push(estimated_hours);
    }
    if (actual_hours !== undefined) {
      updateFields.push('actual_hours = ?');
      updateValues.push(actual_hours);
    }
    if (assigned_to !== undefined) {
      updateFields.push('assigned_to = ?');
      updateValues.push(assigned_to);
    }
    if (due_date !== undefined) {
      updateFields.push('due_date = ?');
      updateValues.push(due_date);
    }
    if (sprint_id !== undefined) {
      updateFields.push('sprint_id = ?');
      updateValues.push(sprint_id);
    }

    if (updateFields.length === 0) {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'No fields to update'
      });
    }

    updateValues.push(taskId);

    // Update task
    await connection.execute(
      `UPDATE tasks SET ${updateFields.join(', ')}, updated_at = CURRENT_TIMESTAMP WHERE id = ?`,
      updateValues
    );

    // Log activity
    await connection.execute(
      'INSERT INTO activity_logs (user_id, action, entity_type, entity_id, details) VALUES (?, ?, ?, ?, ?)',
      [updatedBy, 'update_task', 'task', taskId, JSON.stringify(req.body)]
    );

    await connection.commit();

    // Get updated task details
    const [updatedTask] = await connection.execute(`
      SELECT 
        t.id,
        t.title,
        t.description,
        t.status,
        t.priority,
        t.type,
        t.story_points,
        t.estimated_hours,
        t.actual_hours,
        t.due_date,
        t.created_at,
        t.updated_at,
        t.project_id as project_id,
        t.sprint_id as sprint_id,
        t.assigned_to as assigned_to,
        t.created_by as created_by,
        p.name as project_name,
        s.name as sprint_name,
        assignee.id as assignee_id,
        assignee.first_name as assignee_first_name,
        assignee.last_name as assignee_last_name,
        creator.id as creator_id,
        creator.first_name as creator_first_name,
        creator.last_name as creator_last_name
      FROM tasks t
      LEFT JOIN projects p ON t.project_id = p.id
      LEFT JOIN sprints s ON t.sprint_id = s.id
      LEFT JOIN users assignee ON t.assigned_to = assignee.id
      LEFT JOIN users creator ON t.created_by = creator.id
      WHERE t.id = ?
    `, [taskId]);

    const normalizedRows = Array.isArray(updatedTask) ? updatedTask : [updatedTask];
    const enrichedUpdatedTask = await enrichTasksWithAssignments(normalizedRows);
    const taskResponse = Array.isArray(enrichedUpdatedTask) && enrichedUpdatedTask.length > 0
      ? enrichedUpdatedTask[0]
      : (Array.isArray(normalizedRows) && normalizedRows.length > 0 ? normalizedRows[0] : {});

    res.status(200).json({
      success: true,
      message: 'Task updated successfully',
      data: {
        task: taskResponse
      }
    });
  } catch (error) {
    await connection.rollback();
    console.error('Update task error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update task'
    });
  } finally {
    connection.release();
  }
});

// Update task status
router.patch('/:id/status', authenticateToken, validateId(), handleValidationErrors, async (req, res) => {
  const connection = await getConnection();

  try {
    const taskId = req.params.id;
    const { status } = req.body;

    if (!status) {
      return res.status(400).json({
        success: false,
        message: 'Status is required',
      });
    }

    await connection.beginTransaction();

    const updatedBy = req.user.id;
    const userRole = req.user.role;

    let taskQuery = `
      SELECT t.*, p.owner_id as project_owner_id
      FROM tasks t
      JOIN projects p ON t.project_id = p.id
      WHERE t.id = ?
    `;
    let taskParams = [taskId];

    if (userRole !== 'admin' && userRole !== 'main_admin') {
      taskQuery += ' AND (p.owner_id = ? OR p.id IN (SELECT project_id FROM project_members WHERE user_id = ?) OR t.assigned_to = ?)';
      taskParams.push(updatedBy, updatedBy, updatedBy);
    }

    const [existingTasks] = await connection.execute(taskQuery, taskParams);

    if (!existingTasks || existingTasks.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Task not found or access denied',
      });
    }

    const mappedStatus = mapStatusToDb(status);

    await connection.execute(
      'UPDATE tasks SET status = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
      [mappedStatus, taskId]
    );

    await connection.execute(
      'INSERT INTO activity_logs (user_id, action, entity_type, entity_id, details) VALUES (?, ?, ?, ?, ?)',
      [updatedBy, 'update_task_status', 'task', taskId, JSON.stringify({ status: mappedStatus })]
    );

    await connection.commit();

    const [updatedTask] = await connection.execute(`
      SELECT 
        t.id,
        t.title,
        t.description,
        t.status,
        t.priority,
        t.type,
        t.story_points,
        t.estimated_hours,
        t.actual_hours,
        t.due_date,
        t.created_at,
        t.updated_at,
        t.project_id as project_id,
        t.sprint_id as sprint_id,
        t.assigned_to as assigned_to,
        t.created_by as created_by,
        p.name as project_name,
        s.name as sprint_name,
        assignee.id as assignee_id,
        assignee.first_name as assignee_first_name,
        assignee.last_name as assignee_last_name,
        creator.id as creator_id,
        creator.first_name as creator_first_name,
        creator.last_name as creator_last_name
      FROM tasks t
      LEFT JOIN projects p ON t.project_id = p.id
      LEFT JOIN sprints s ON t.sprint_id = s.id
      LEFT JOIN users assignee ON t.assigned_to = assignee.id
      LEFT JOIN users creator ON t.created_by = creator.id
      WHERE t.id = ?
    `, [taskId]);

    const normalizedRows = Array.isArray(updatedTask) ? updatedTask : [updatedTask];
    const enrichedUpdatedTask = await enrichTasksWithAssignments(normalizedRows);
    const taskResponse = Array.isArray(enrichedUpdatedTask) && enrichedUpdatedTask.length > 0
      ? enrichedUpdatedTask[0]
      : (Array.isArray(normalizedRows) && normalizedRows.length > 0 ? normalizedRows[0] : {});

    res.status(200).json({
      success: true,
      message: 'Task status updated successfully',
      data: {
        task: taskResponse,
      },
    });
  } catch (error) {
    await connection.rollback();
    console.error('Update task status error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update task status',
    });
  } finally {
    connection.release();
  }
});

// Update task priority
router.patch('/:id/priority', authenticateToken, validateId(), handleValidationErrors, async (req, res) => {
  const connection = await getConnection();

  try {
    const taskId = req.params.id;
    const { priority } = req.body;

    if (!priority) {
      return res.status(400).json({
        success: false,
        message: 'Priority is required',
      });
    }

    await connection.beginTransaction();

    const updatedBy = req.user.id;
    const userRole = req.user.role;

    let taskQuery = `
      SELECT t.*, p.owner_id as project_owner_id
      FROM tasks t
      JOIN projects p ON t.project_id = p.id
      WHERE t.id = ?
    `;
    let taskParams = [taskId];

    if (userRole !== 'admin' && userRole !== 'main_admin') {
      taskQuery += ' AND (p.owner_id = ? OR p.id IN (SELECT project_id FROM project_members WHERE user_id = ?) OR t.assigned_to = ?)';
      taskParams.push(updatedBy, updatedBy, updatedBy);
    }

    const [existingTasks] = await connection.execute(taskQuery, taskParams);

    if (!existingTasks || existingTasks.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Task not found or access denied',
      });
    }

    const mappedPriority = mapPriorityToDb(priority);

    await connection.execute(
      'UPDATE tasks SET priority = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
      [mappedPriority, taskId]
    );

    await connection.execute(
      'INSERT INTO activity_logs (user_id, action, entity_type, entity_id, details) VALUES (?, ?, ?, ?, ?)',
      [updatedBy, 'update_task_priority', 'task', taskId, JSON.stringify({ priority: mappedPriority })]
    );

    await connection.commit();

    const [updatedTask] = await connection.execute(`
      SELECT 
        t.id,
        t.title,
        t.description,
        t.status,
        t.priority,
        t.type,
        t.story_points,
        t.estimated_hours,
        t.actual_hours,
        t.due_date,
        t.created_at,
        t.updated_at,
        t.project_id as project_id,
        t.sprint_id as sprint_id,
        t.assigned_to as assigned_to,
        t.created_by as created_by,
        p.name as project_name,
        s.name as sprint_name,
        assignee.id as assignee_id,
        assignee.first_name as assignee_first_name,
        assignee.last_name as assignee_last_name,
        creator.id as creator_id,
        creator.first_name as creator_first_name,
        creator.last_name as creator_last_name
      FROM tasks t
      LEFT JOIN projects p ON t.project_id = p.id
      LEFT JOIN sprints s ON t.sprint_id = s.id
      LEFT JOIN users assignee ON t.assigned_to = assignee.id
      LEFT JOIN users creator ON t.created_by = creator.id
      WHERE t.id = ?
    `, [taskId]);

    const normalizedRows = Array.isArray(updatedTask) ? updatedTask : [updatedTask];
    const enrichedUpdatedTask = await enrichTasksWithAssignments(normalizedRows);
    const taskResponse = Array.isArray(enrichedUpdatedTask) && enrichedUpdatedTask.length > 0
      ? enrichedUpdatedTask[0]
      : (Array.isArray(normalizedRows) && normalizedRows.length > 0 ? normalizedRows[0] : {});

    res.status(200).json({
      success: true,
      message: 'Task priority updated successfully',
      data: {
        task: taskResponse,
      },
    });
  } catch (error) {
    await connection.rollback();
    console.error('Update task priority error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update task priority',
    });
  } finally {
    connection.release();
  }
});

// Add task assignee (additional)
router.post('/:id/assignees', authenticateToken, validateId(), handleValidationErrors, async (req, res) => {
  const connection = await getConnection();

  try {
    const taskId = req.params.id;
    const { user_id } = req.body;

    if (!user_id) {
      return res.status(400).json({
        success: false,
        message: 'user_id is required',
      });
    }

    await connection.beginTransaction();

    const actingUserId = req.user.id;
    const userRole = req.user.role;

    let taskQuery = `
      SELECT t.*, p.owner_id as project_owner_id
      FROM tasks t
      JOIN projects p ON t.project_id = p.id
      WHERE t.id = ?
    `;
    let taskParams = [taskId];

    if (userRole !== 'admin' && userRole !== 'main_admin') {
      taskQuery += ' AND (p.owner_id = ? OR p.id IN (SELECT project_id FROM project_members WHERE user_id = ?) OR t.assigned_to = ?)';
      taskParams.push(actingUserId, actingUserId, actingUserId);
    }

    const [existingTasks] = await connection.execute(taskQuery, taskParams);

    if (!existingTasks || existingTasks.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Task not found or access denied',
      });
    }

    const existingTask = existingTasks[0];

    // Verify user exists and has access to the project
    const [eligibleUsers] = await connection.execute(
      'SELECT id FROM users WHERE id = ? AND status = "active" AND (id IN (SELECT user_id FROM project_members WHERE project_id = ?) OR id IN (SELECT owner_id FROM projects WHERE id = ?))',
      [user_id, existingTask.project_id, existingTask.project_id]
    );

    if (!eligibleUsers || eligibleUsers.length === 0) {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'User not found or does not have access to this project',
      });
    }

    if (existingTask.assigned_to && existingTask.assigned_to.toString() === user_id.toString()) {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'User is already the primary assignee for this task',
      });
    }

    const [duplicateCheck] = await connection.execute(
      'SELECT id FROM task_assignments WHERE task_id = ? AND user_id = ?',
      [taskId, user_id]
    );

    if (duplicateCheck && duplicateCheck.length > 0) {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'User is already assigned to this task',
      });
    }

    await connection.execute(
      'INSERT INTO task_assignments (task_id, user_id, assigned_by) VALUES (?, ?, ?)',
      [taskId, user_id, actingUserId]
    );

    if (!existingTask.assigned_to) {
      await connection.execute(
        'UPDATE tasks SET assigned_to = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
        [user_id, taskId]
      );
    }

    await connection.execute(
      'INSERT INTO activity_logs (user_id, action, entity_type, entity_id, details) VALUES (?, ?, ?, ?, ?)',
      [actingUserId, 'assign_task_user', 'task', taskId, JSON.stringify({ user_id })]
    );

    await connection.commit();

    const [updatedTask] = await connection.execute(`
      SELECT 
        t.id,
        t.title,
        t.description,
        t.status,
        t.priority,
        t.type,
        t.story_points,
        t.estimated_hours,
        t.actual_hours,
        t.due_date,
        t.created_at,
        t.updated_at,
        t.project_id as project_id,
        t.sprint_id as sprint_id,
        t.assigned_to as assigned_to,
        t.created_by as created_by,
        p.name as project_name,
        s.name as sprint_name,
        assignee.id as assignee_id,
        assignee.first_name as assignee_first_name,
        assignee.last_name as assignee_last_name,
        creator.id as creator_id,
        creator.first_name as creator_first_name,
        creator.last_name as creator_last_name
      FROM tasks t
      LEFT JOIN projects p ON t.project_id = p.id
      LEFT JOIN sprints s ON t.sprint_id = s.id
      LEFT JOIN users assignee ON t.assigned_to = assignee.id
      LEFT JOIN users creator ON t.created_by = creator.id
      WHERE t.id = ?
    `, [taskId]);

    const normalizedRows = Array.isArray(updatedTask) ? updatedTask : [updatedTask];
    const enriched = await enrichTasksWithAssignments(normalizedRows);
    const taskResponse = Array.isArray(enriched) && enriched.length > 0
      ? enriched[0]
      : (Array.isArray(normalizedRows) && normalizedRows.length > 0 ? normalizedRows[0] : {});

    // Send notification to assigned user
    if (user_id && taskResponse.title) {
      const [assigneeResult] = await connection.execute(
        'SELECT first_name, last_name FROM users WHERE id = ?',
        [user_id]
      );
      
      if (assigneeResult && assigneeResult.length > 0) {
        const assignee = assigneeResult[0];
        const assigneeName = `${assignee.first_name || ''} ${assignee.last_name || ''}`.trim() || 'User';
        const creatorName = `${taskResponse.creator_first_name || ''} ${taskResponse.creator_last_name || ''}`.trim() || 'Someone';
        
        await notificationService.sendNotificationToUser(
          user_id,
          'New Task Assigned',
          `${creatorName} assigned you to task: ${taskResponse.title}`,
          {
            task_id: taskId.toString(),
            task_title: taskResponse.title,
            type: 'task_assigned'
          }
        );
      }
    }

    res.status(200).json({
      success: true,
      message: 'Assignee added successfully',
      data: {
        task: taskResponse,
      },
    });
  } catch (error) {
    await connection.rollback();
    console.error('Add task assignee error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to add assignee',
    });
  } finally {
    connection.release();
  }
});

// Remove additional task assignee
router.delete('/:id/assignees/:userId', authenticateToken, validateId(), handleValidationErrors, async (req, res) => {
  const connection = await getConnection();

  try {
    const taskId = req.params.id;
    const userId = req.params.userId;

    await connection.beginTransaction();

    const actingUserId = req.user.id;
    const userRole = req.user.role;

    let taskQuery = `
      SELECT t.*, p.owner_id as project_owner_id
      FROM tasks t
      JOIN projects p ON t.project_id = p.id
      WHERE t.id = ?
    `;
    let taskParams = [taskId];

    if (userRole !== 'admin' && userRole !== 'main_admin') {
      taskQuery += ' AND (p.owner_id = ? OR p.id IN (SELECT project_id FROM project_members WHERE user_id = ?) OR t.assigned_to = ?)';
      taskParams.push(actingUserId, actingUserId, actingUserId);
    }

    const [existingTasks] = await connection.execute(taskQuery, taskParams);

    if (!existingTasks || existingTasks.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Task not found or access denied',
      });
    }

    const existingTask = existingTasks[0];

    await connection.execute(
      'DELETE FROM task_assignments WHERE task_id = ? AND user_id = ?',
      [taskId, userId]
    );

    if (existingTask.assigned_to && existingTask.assigned_to.toString() === userId.toString()) {
      const [nextAssignee] = await connection.execute(
        'SELECT user_id FROM task_assignments WHERE task_id = ? ORDER BY assigned_at ASC LIMIT 1',
        [taskId]
      );

      const newPrimary = nextAssignee && nextAssignee.length > 0 ? nextAssignee[0].user_id : null;

      await connection.execute(
        'UPDATE tasks SET assigned_to = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
        [newPrimary, taskId]
      );
    }

    await connection.execute(
      'INSERT INTO activity_logs (user_id, action, entity_type, entity_id, details) VALUES (?, ?, ?, ?, ?)',
      [actingUserId, 'unassign_task_user', 'task', taskId, JSON.stringify({ user_id: userId })]
    );

    await connection.commit();

    const [updatedTask] = await connection.execute(`
      SELECT 
        t.id,
        t.title,
        t.description,
        t.status,
        t.priority,
        t.type,
        t.story_points,
        t.estimated_hours,
        t.actual_hours,
        t.due_date,
        t.created_at,
        t.updated_at,
        t.project_id as project_id,
        t.sprint_id as sprint_id,
        t.assigned_to as assigned_to,
        t.created_by as created_by,
        p.name as project_name,
        s.name as sprint_name,
        assignee.id as assignee_id,
        assignee.first_name as assignee_first_name,
        assignee.last_name as assignee_last_name,
        creator.id as creator_id,
        creator.first_name as creator_first_name,
        creator.last_name as creator_last_name
      FROM tasks t
      LEFT JOIN projects p ON t.project_id = p.id
      LEFT JOIN sprints s ON t.sprint_id = s.id
      LEFT JOIN users assignee ON t.assigned_to = assignee.id
      LEFT JOIN users creator ON t.created_by = creator.id
      WHERE t.id = ?
    `, [taskId]);

    const normalizedRows = Array.isArray(updatedTask) ? updatedTask : [updatedTask];
    const enriched = await enrichTasksWithAssignments(normalizedRows);
    const taskResponse = Array.isArray(enriched) && enriched.length > 0
      ? enriched[0]
      : (Array.isArray(normalizedRows) && normalizedRows.length > 0 ? normalizedRows[0] : {});

    res.status(200).json({
      success: true,
      message: 'Assignee removed successfully',
      data: {
        task: taskResponse,
      },
    });
  } catch (error) {
    await connection.rollback();
    console.error('Remove task assignee error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to remove assignee',
    });
  } finally {
    connection.release();
  }
});

// Set primary assignee
router.patch('/:id/assign', authenticateToken, validateId(), handleValidationErrors, async (req, res) => {
  const connection = await getConnection();

  try {
    const taskId = req.params.id;
    const { assigned_to } = req.body;

    if (!assigned_to) {
      return res.status(400).json({
        success: false,
        message: 'assigned_to is required',
      });
    }

    await connection.beginTransaction();

    const actingUserId = req.user.id;
    const userRole = req.user.role;

    let taskQuery = `
      SELECT t.*, p.owner_id as project_owner_id
      FROM tasks t
      JOIN projects p ON t.project_id = p.id
      WHERE t.id = ?
    `;
    let taskParams = [taskId];

    if (userRole !== 'admin' && userRole !== 'main_admin') {
      taskQuery += ' AND (p.owner_id = ? OR p.id IN (SELECT project_id FROM project_members WHERE user_id = ?) OR t.assigned_to = ?)';
      taskParams.push(actingUserId, actingUserId, actingUserId);
    }

    const [existingTasks] = await connection.execute(taskQuery, taskParams);

    if (!existingTasks || existingTasks.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Task not found or access denied',
      });
    }

    const existingTask = existingTasks[0];

    const [eligibleUsers] = await connection.execute(
      'SELECT id FROM users WHERE id = ? AND status = "active" AND (id IN (SELECT user_id FROM project_members WHERE project_id = ?) OR id IN (SELECT owner_id FROM projects WHERE id = ?))',
      [assigned_to, existingTask.project_id, existingTask.project_id]
    );

    if (!eligibleUsers || eligibleUsers.length === 0) {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'User not found or does not have access to this project',
      });
    }

    await connection.execute(
      'UPDATE tasks SET assigned_to = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
      [assigned_to, taskId]
    );

    await connection.execute(
      'INSERT IGNORE INTO task_assignments (task_id, user_id, assigned_by) VALUES (?, ?, ?)',
      [taskId, assigned_to, actingUserId]
    );

    await connection.execute(
      'INSERT INTO activity_logs (user_id, action, entity_type, entity_id, details) VALUES (?, ?, ?, ?, ?)',
      [actingUserId, 'assign_task_primary', 'task', taskId, JSON.stringify({ assigned_to })]
    );

    await connection.commit();

    const [updatedTask] = await connection.execute(`
      SELECT 
        t.id,
        t.title,
        t.description,
        t.status,
        t.priority,
        t.type,
        t.story_points,
        t.estimated_hours,
        t.actual_hours,
        t.due_date,
        t.created_at,
        t.updated_at,
        t.project_id as project_id,
        t.sprint_id as sprint_id,
        t.assigned_to as assigned_to,
        t.created_by as created_by,
        p.name as project_name,
        s.name as sprint_name,
        assignee.id as assignee_id,
        assignee.first_name as assignee_first_name,
        assignee.last_name as assignee_last_name,
        creator.id as creator_id,
        creator.first_name as creator_first_name,
        creator.last_name as creator_last_name
      FROM tasks t
      LEFT JOIN projects p ON t.project_id = p.id
      LEFT JOIN sprints s ON t.sprint_id = s.id
      LEFT JOIN users assignee ON t.assigned_to = assignee.id
      LEFT JOIN users creator ON t.created_by = creator.id
      WHERE t.id = ?
    `, [taskId]);

    const normalizedRows = Array.isArray(updatedTask) ? updatedTask : [updatedTask];
    const enriched = await enrichTasksWithAssignments(normalizedRows);
    const taskResponse = Array.isArray(enriched) && enriched.length > 0
      ? enriched[0]
      : (Array.isArray(normalizedRows) && normalizedRows.length > 0 ? normalizedRows[0] : {});

    // Send notification to assigned user
    if (assigned_to && taskResponse.title) {
      const [assigneeResult] = await connection.execute(
        'SELECT first_name, last_name FROM users WHERE id = ?',
        [assigned_to]
      );
      
      if (assigneeResult && assigneeResult.length > 0) {
        const assignee = assigneeResult[0];
        const assigneeName = `${assignee.first_name || ''} ${assignee.last_name || ''}`.trim() || 'User';
        const creatorName = `${taskResponse.creator_first_name || ''} ${taskResponse.creator_last_name || ''}`.trim() || 'Someone';
        
        await notificationService.sendNotificationToUser(
          assigned_to,
          'Task Assigned',
          `${creatorName} assigned you to task: ${taskResponse.title}`,
          {
            task_id: taskId.toString(),
            task_title: taskResponse.title,
            type: 'task_assigned'
          }
        );
      }
    }

    res.status(200).json({
      success: true,
      message: 'Primary assignee updated',
      data: {
        task: taskResponse,
      },
    });
  } catch (error) {
    await connection.rollback();
    console.error('Assign task error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to assign task',
    });
  } finally {
    connection.release();
  }
});

// Clear primary assignee
router.patch('/:id/unassign', authenticateToken, validateId(), handleValidationErrors, async (req, res) => {
  const connection = await getConnection();

  try {
    const taskId = req.params.id;

    await connection.beginTransaction();

    const actingUserId = req.user.id;
    const userRole = req.user.role;

    let taskQuery = `
      SELECT t.*, p.owner_id as project_owner_id
      FROM tasks t
      JOIN projects p ON t.project_id = p.id
      WHERE t.id = ?
    `;
    let taskParams = [taskId];

    if (userRole !== 'admin' && userRole !== 'main_admin') {
      taskQuery += ' AND (p.owner_id = ? OR p.id IN (SELECT project_id FROM project_members WHERE user_id = ?) OR t.assigned_to = ?)';
      taskParams.push(actingUserId, actingUserId, actingUserId);
    }

    const [existingTasks] = await connection.execute(taskQuery, taskParams);

    if (!existingTasks || existingTasks.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Task not found or access denied',
      });
    }

    await connection.execute(
      'UPDATE tasks SET assigned_to = NULL, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
      [taskId]
    );

    await connection.execute(
      'INSERT INTO activity_logs (user_id, action, entity_type, entity_id, details) VALUES (?, ?, ?, ?, ?)',
      [actingUserId, 'unassign_task_primary', 'task', taskId, JSON.stringify({ assigned_to: null })]
    );

    await connection.commit();

    const [updatedTask] = await connection.execute(`
      SELECT 
        t.id,
        t.title,
        t.description,
        t.status,
        t.priority,
        t.type,
        t.story_points,
        t.estimated_hours,
        t.actual_hours,
        t.due_date,
        t.created_at,
        t.updated_at,
        t.project_id as project_id,
        t.sprint_id as sprint_id,
        t.assigned_to as assigned_to,
        t.created_by as created_by,
        p.name as project_name,
        s.name as sprint_name,
        assignee.id as assignee_id,
        assignee.first_name as assignee_first_name,
        assignee.last_name as assignee_last_name,
        creator.id as creator_id,
        creator.first_name as creator_first_name,
        creator.last_name as creator_last_name
      FROM tasks t
      LEFT JOIN projects p ON t.project_id = p.id
      LEFT JOIN sprints s ON t.sprint_id = s.id
      LEFT JOIN users assignee ON t.assigned_to = assignee.id
      LEFT JOIN users creator ON t.created_by = creator.id
      WHERE t.id = ?
    `, [taskId]);

    const normalizedRows = Array.isArray(updatedTask) ? updatedTask : [updatedTask];
    const enriched = await enrichTasksWithAssignments(normalizedRows);
    const taskResponse = Array.isArray(enriched) && enriched.length > 0
      ? enriched[0]
      : (Array.isArray(normalizedRows) && normalizedRows.length > 0 ? normalizedRows[0] : {});

    res.status(200).json({
      success: true,
      message: 'Primary assignee cleared',
      data: {
        task: taskResponse,
      },
    });
  } catch (error) {
    await connection.rollback();
    console.error('Unassign task error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to unassign task',
    });
  } finally {
    connection.release();
  }
});

// Delete task
router.delete('/:id', authenticateToken, validateId(), handleValidationErrors, async (req, res) => {
  const connection = await getConnection();
  
  try {
    await connection.beginTransaction();

    const taskId = req.params.id;
    const deletedBy = req.user.id;
    const userRole = req.user.role;

    // Check if task exists and user has permission to delete
    let taskQuery = `
      SELECT t.*, p.owner_id as project_owner_id
      FROM tasks t
      JOIN projects p ON t.project_id = p.id
      WHERE t.id = ?
    `;
    let taskParams = [taskId];

    // Only admins, project owners, or task creators can delete tasks
    if (userRole !== 'admin' && userRole !== 'main_admin') {
      taskQuery += ' AND (p.owner_id = ? OR t.created_by = ?)';
      taskParams.push(deletedBy, deletedBy);
    }

    const [existingTasks] = await connection.execute(taskQuery, taskParams);

    if (existingTasks.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Task not found or insufficient permissions'
      });
    }

    // Delete task (this will cascade to related tables)
    await connection.execute(
      'DELETE FROM tasks WHERE id = ?',
      [taskId]
    );

    // Log activity
    await connection.execute(
      'INSERT INTO activity_logs (user_id, action, entity_type, entity_id, details) VALUES (?, ?, ?, ?, ?)',
      [deletedBy, 'delete_task', 'task', taskId, JSON.stringify({ task_id: taskId })]
    );

    await connection.commit();

    res.status(200).json({
      success: true,
      message: 'Task deleted successfully'
    });
  } catch (error) {
    await connection.rollback();
    console.error('Delete task error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to delete task'
    });
  } finally {
    connection.release();
  }
});

// Add comment to task
router.post('/:id/comments', authenticateToken, validateId(), handleValidationErrors, async (req, res) => {
  const connection = await getConnection();
  
  try {
    await connection.beginTransaction();

    const taskId = req.params.id;
    const { comment } = req.body;
    const userId = req.user.id;
    const userRole = req.user.role;

    if (!comment || comment.trim().length === 0) {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'Comment is required'
      });
    }

    // Check if task exists and user has access
    let taskQuery = `
      SELECT t.id
      FROM tasks t
      JOIN projects p ON t.project_id = p.id
      WHERE t.id = ?
    `;
    let taskParams = [taskId];

    if (userRole !== 'admin' && userRole !== 'main_admin') {
      taskQuery += ' AND (p.owner_id = ? OR p.id IN (SELECT project_id FROM project_members WHERE user_id = ?) OR t.assigned_to = ?)';
      taskParams.push(userId, userId, userId);
    }

    const [taskExists] = await connection.execute(taskQuery, taskParams);

    if (taskExists.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Task not found or access denied'
      });
    }

    // Add comment
    const [result] = await connection.execute(
      'INSERT INTO task_comments (task_id, user_id, comment) VALUES (?, ?, ?)',
      [taskId, userId, comment.trim()]
    );

    const commentId = result.insertId;

    // Log activity
    await connection.execute(
      'INSERT INTO activity_logs (user_id, action, entity_type, entity_id, details) VALUES (?, ?, ?, ?, ?)',
      [userId, 'add_task_comment', 'task', taskId, JSON.stringify({ comment_id: commentId })]
    );

    await connection.commit();

    // Get comment details
    const [newComment] = await connection.execute(`
      SELECT 
        tc.id, tc.comment, tc.created_at,
        u.first_name, u.last_name
      FROM task_comments tc
      JOIN users u ON tc.user_id = u.id
      WHERE tc.id = ?
    `, [commentId]);

    res.status(201).json({
      success: true,
      message: 'Comment added successfully',
      data: {
        comment: newComment[0]
      }
    });
  } catch (error) {
    await connection.rollback();
    console.error('Add task comment error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to add comment'
    });
  } finally {
    connection.release();
  }
});

module.exports = router;
