/**
 * Test script for subscription flow
 * This script tests the subscription functionality of the backend
 */

const { executeQuery } = require('../config/database');

async function testSubscriptionFlow() {
  console.log('🧪 Testing Subscription Flow...\n');

  try {
    // Test 1: Check if subscriptions table exists
    console.log('📋 Test 1: Checking if subscriptions table exists...');
    const tableCheck = await executeQuery(`
      SELECT table_name 
      FROM information_schema.tables 
      WHERE table_schema = DATABASE() 
      AND table_name = 'subscriptions'
    `);
    
    if (tableCheck.length > 0) {
      console.log('✅ Subscriptions table exists');
    } else {
      console.log('❌ Subscriptions table does not exist');
      return;
    }

    // Test 2: Check if users table has subscription fields
    console.log('\n📋 Test 2: Checking if users table has subscription fields...');
    const columnCheck = await executeQuery(`
      SELECT column_name 
      FROM information_schema.columns 
      WHERE table_schema = DATABASE() 
      AND table_name = 'users' 
      AND column_name IN ('subscription_status', 'subscription_end_date')
    `);
    
    if (columnCheck.length >= 2) {
      console.log('✅ Users table has subscription fields');
    } else {
      console.log('❌ Users table missing subscription fields');
      return;
    }

    // Test 3: Insert test subscription
    console.log('\n📋 Test 3: Inserting test subscription...');
    const testUserId = 1; // Assuming user with ID 1 exists
    
    // Format dates properly for MySQL
    const now = new Date();
    const startDate = now.toISOString().slice(0, 19).replace('T', ' ');
    const endDate = new Date(now.getTime() + 30 * 24 * 60 * 60 * 1000)
      .toISOString().slice(0, 19).replace('T', ' ');
    
    const testSubscription = {
      user_id: testUserId,
      platform: 'ios',
      product_id: 'com.projectmanager.premium.monthly',
      transaction_id: 'test_transaction_' + Date.now(),
      receipt_data: 'test_receipt_data',
      status: 'active',
      start_date: startDate,
      end_date: endDate,
    };

    const insertResult = await executeQuery(`
      INSERT INTO subscriptions 
      (user_id, platform, product_id, transaction_id, receipt_data, status, start_date, end_date) 
      VALUES (?, ?, ?, ?, ?, ?, ?, ?)
    `, [
      testSubscription.user_id,
      testSubscription.platform,
      testSubscription.product_id,
      testSubscription.transaction_id,
      testSubscription.receipt_data,
      testSubscription.status,
      testSubscription.start_date,
      testSubscription.end_date
    ]);

    if (insertResult.insertId) {
      console.log('✅ Test subscription inserted successfully');
      console.log(`📝 Subscription ID: ${insertResult.insertId}`);
    } else {
      console.log('❌ Failed to insert test subscription');
      return;
    }

    // Test 4: Update user subscription status
    console.log('\n📋 Test 4: Updating user subscription status...');
    const updateResult = await executeQuery(`
      UPDATE users 
      SET subscription_status = ?, subscription_end_date = ? 
      WHERE id = ?
    `, [
      testSubscription.status,
      testSubscription.end_date,
      testSubscription.user_id
    ]);

    if (updateResult.affectedRows > 0) {
      console.log('✅ User subscription status updated successfully');
    } else {
      console.log('❌ Failed to update user subscription status');
      return;
    }

    // Test 5: Retrieve subscription
    console.log('\n📋 Test 5: Retrieving subscription...');
    const selectResult = await executeQuery(`
      SELECT * FROM subscriptions 
      WHERE id = ?
    `, [insertResult.insertId]);

    if (selectResult.length > 0) {
      console.log('✅ Subscription retrieved successfully');
      console.log(`📝 Subscription: ${JSON.stringify(selectResult[0], null, 2)}`);
    } else {
      console.log('❌ Failed to retrieve subscription');
      return;
    }

    // Test 6: Clean up test data
    console.log('\n📋 Test 6: Cleaning up test data...');
    const deleteResult = await executeQuery(`
      DELETE FROM subscriptions 
      WHERE id = ?
    `, [insertResult.insertId]);

    if (deleteResult.affectedRows > 0) {
      console.log('✅ Test data cleaned up successfully');
    } else {
      console.log('❌ Failed to clean up test data');
    }

    console.log('\n🎉 All tests completed successfully!');
    console.log('\n📋 Subscription Flow Test Summary:');
    console.log('✅ Subscriptions table exists');
    console.log('✅ Users table has subscription fields');
    console.log('✅ Subscription creation works');
    console.log('✅ User subscription status update works');
    console.log('✅ Subscription retrieval works');
    console.log('✅ Test data cleanup works');

  } catch (error) {
    console.error('❌ Test failed with error:', error);
    process.exit(1);
  }
}

// Run the test
if (require.main === module) {
  testSubscriptionFlow().then(() => {
    process.exit(0);
  });
}

module.exports = { testSubscriptionFlow };