const admin = require('firebase-admin');
const { executeQuery } = require('../config/database');

// Initialize Firebase Admin SDK if not already initialized
if (!admin.apps.length) {
  try {
    admin.initializeApp({
      credential: admin.credential.applicationDefault()
    });
    console.log('Firebase Admin SDK initialized for notifications');
  } catch (error) {
    console.log('Firebase Admin SDK not initialized for notifications:', error.message);
  }
}

class NotificationService {
  /**
   * Send push notification to a specific user
   * @param {number} userId - The ID of the user to notify
   * @param {string} title - Notification title
   * @param {string} body - Notification body
   * @param {object} data - Additional data to send with notification
   */
  async sendNotificationToUser(userId, title, body, data = {}) {
    try {
      // Get user's FCM token from database
      const userQuery = 'SELECT fcm_token FROM users WHERE id = ? AND fcm_token IS NOT NULL AND status = "active"';
      const users = await executeQuery(userQuery, [userId]);
      
      if (users.length === 0) {
        console.log(`No FCM token found for user ${userId} or user is inactive`);
        return { success: false, message: 'User not found or no FCM token available' };
      }
      
      const fcmToken = users[0].fcm_token;
      
      // Prepare notification payload
      const message = {
        token: fcmToken,
        notification: {
          title: title,
          body: body,
        },
        data: {
          ...data,
          click_action: 'FLUTTER_NOTIFICATION_CLICK',
          user_id: userId.toString(),
        }
      };
      
      // Send notification
      const response = await admin.messaging().send(message);
      console.log('Successfully sent notification:', response);
      
      return { success: true, messageId: response };
    } catch (error) {
      console.error('Error sending notification:', error);
      return { success: false, message: error.message };
    }
  }
  
  /**
   * Send push notification to multiple users
   * @param {number[]} userIds - Array of user IDs to notify
   * @param {string} title - Notification title
   * @param {string} body - Notification body
   * @param {object} data - Additional data to send with notification
   */
  async sendNotificationToUsers(userIds, title, body, data = {}) {
    try {
      if (!Array.isArray(userIds) || userIds.length === 0) {
        return { success: false, message: 'Invalid user IDs provided' };
      }
      
      // Get FCM tokens for all users
      const placeholders = userIds.map(() => '?').join(',');
      const userQuery = `SELECT id, fcm_token FROM users WHERE id IN (${placeholders}) AND fcm_token IS NOT NULL AND status = "active"`;
      const users = await executeQuery(userQuery, userIds);
      
      if (users.length === 0) {
        console.log('No users found with valid FCM tokens');
        return { success: false, message: 'No users found with valid FCM tokens' };
      }
      
      // Send notifications to each user
      const results = [];
      for (const user of users) {
        try {
          const message = {
            token: user.fcm_token,
            notification: {
              title: title,
              body: body,
            },
            data: {
              ...data,
              click_action: 'FLUTTER_NOTIFICATION_CLICK',
              user_id: user.id.toString(),
            }
          };
          
          const response = await admin.messaging().send(message);
          results.push({ userId: user.id, success: true, messageId: response });
        } catch (error) {
          console.error(`Error sending notification to user ${user.id}:`, error);
          results.push({ userId: user.id, success: false, message: error.message });
        }
      }
      
      return { success: true, results };
    } catch (error) {
      console.error('Error sending notifications to users:', error);
      return { success: false, message: error.message };
    }
  }
  
  /**
   * Update user's FCM token
   * @param {number} userId - The ID of the user
   * @param {string} fcmToken - The FCM token to store
   */
  async updateUserFcmToken(userId, fcmToken) {
    try {
      await executeQuery(
        'UPDATE users SET fcm_token = ? WHERE id = ?',
        [fcmToken, userId]
      );
      
      console.log(`Updated FCM token for user ${userId}`);
      return { success: true };
    } catch (error) {
      console.error('Error updating user FCM token:', error);
      return { success: false, message: error.message };
    }
  }
  
  /**
   * Remove user's FCM token (e.g., when logging out)
   * @param {number} userId - The ID of the user
   */
  async removeUserFcmToken(userId) {
    try {
      await executeQuery(
        'UPDATE users SET fcm_token = NULL WHERE id = ?',
        [userId]
      );
      
      console.log(`Removed FCM token for user ${userId}`);
      return { success: true };
    } catch (error) {
      console.error('Error removing user FCM token:', error);
      return { success: false, message: error.message };
    }
  }
}

module.exports = new NotificationService();